#!/usr/bin/env python

#   This file is part of Cupydon.
#
#    Cupydon  is free software: you can redistribute it and/or modify
#    it under the terms of the GNU General Public License as published by
#    the Free Software Foundation, either version 3 of the License, or
#    (at your option) any later version.
#
#    Cupydon is distributed in the hope that it will be useful,
#    but WITHOUT ANY WARRANTY; without even the implied warranty of
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#    GNU General Public License for more details.
#
#    You should have received a copy of the GNU General Public License
#    along with Cupydon.  If not, see <http://www.gnu.org/licenses/>
#
#    
#    Copyright(c):    ASGA, Gocad Consortium 2009-2013
#

app_name = u'Qupydon'
app_version = u'0.99.0'

from PyQt4 import QtCore, QtGui, QtSvg
import sys, os
Qt = QtCore.Qt
app = QtGui.QApplication(sys.argv)
from optedit import OptEdit, OEInt, OEFloat
debug = False
#debug=True

_basedir = os.path.abspath(os.path.normpath(os.path.dirname(sys.argv[0])))
defref = os.path.join(_basedir, 'stdref.txt')
if debug :
    defref = os.path.join(_basedir, 'stdref2.txt')

from dtwlib import InFile, R2DTW, OutRes


class BaseGView(QtGui.QGraphicsView):
    def __init__(self, main,parent):
        QtGui.QGraphicsView.__init__(self, parent)
        self.main = main
        self._scn = QtGui.QGraphicsScene(self)
        self.setScene(self._scn)
        self.setMouseTracking(True)
        self.init()
        
    def mouseMoveEvent(self, evt):
        QtGui.QGraphicsView.mouseMoveEvent(self, evt)
        p = self.mapToScene(evt.pos())
        txt = self.pos_label(p.x(), p.y())
        self.write_status(txt)
        
    def pos_label(self, x, y):
        return u'%f,%f'%(x, y)
        
        
    def write_status(self, txt=''):
        self.main.mouse_status.setText(txt)
        
    def get_res(self): return self.main.get_cur_res()
    def get_comp(self): return self.main.get_comp()
    def get_ref(self): return self.main.get_ref()
    def get_data(self): return self.main.get_data()
        
    def init(self):
        pass


    def svg_output(self, filename):
        gen = QtSvg.QSvgGenerator();
 
        gen.setFileName( "test.svg" )
        gen.setSize(QtCore.QSize(2000, 2000))
        gen.setViewBox(QtCore.QRectF(0, 0, 2000,  2000))
        print self._scn. sceneRect()
        gen.setTitle("SVG Generator Example Drawing")
        gen.setDescription("An SVG drawing created by the SVG Generator "
                                "Example provided with Qt.")
 
        self.render( QtGui.QPainter( gen ))
        


class DTWView(BaseGView):
    def resizeEvent (self, evt ) :
        if not self._anti_resize_loop:
            self._anti_resize_loop = True
            self.zoom()
            self._anti_resize_loop = False
        QtGui.QGraphicsView.resizeEvent(self, evt)
        
    def contextMenuEvent(self, event):
        self.menu.exec_(event.globalPos());   
    
    def init(self):
        self._anti_resize_loop = False
        self._scene_size = (0, 100,0, 100, 0, 100 )

        
        self._cur_zoom = 0
        m = self.menu = QtGui.QMenu(self)
        from functools import  partial
        m.addAction("Best", partial(self.zoom, 0))
        m.addAction("Full", partial(self.zoom, 1))
        m.addAction("Zoom *2", partial(self.zoom, 2))
        m.addAction("Zoom *4", partial(self.zoom, 4))
        m.addAction("Zoom *8", partial(self.zoom, 8))
        m.addAction("Zoom *16", partial(self.zoom, 16))
        m.addAction("Zoom *32", partial(self.zoom, 32))
        m.addAction("Zoom *64", partial(self.zoom, 64))
        
        
        
    def zoom(self, zf = None, recenter = True):
        if zf != None : self._cur_zoom = zf
        x, w, y, h, by, bh = self._scene_size
        
        
        if self._cur_zoom == 0 :
            self.fitInView(QtCore.QRectF(x, by, w, bh))
        elif self._cur_zoom == 1 :
            self.fitInView(QtCore.QRectF(x, y, w, h))

        else :
                center = self.mapToScene(self.rect().center())
                self.fitInView(QtCore.QRectF(x, y, w, h/self._cur_zoom))
                if recenter : self.centerOn(center)
        
        
        
    def redraw(self):
        ymul = 100.
        scn =self._scn = QtGui.QGraphicsScene(self)
        self.setScene(self._scn)
        
        ref = self.get_ref()
        self._scene_size = (0, 100,0, 100, 0, 100 )

        if not ref : return
        
        data = self.get_data()
        
        brushes  = (QtGui.QBrush(), QtGui.QBrush(Qt.black))

        #ref_start = ref.prof[0]
        def map_y(i0, i1): 
            ref0 = ref.prof[i0]
            refl  = ref.prof[i1] -ref0
            
            data0 = data.prof[0]
            datal = data.prof[-1] -data0
            return map (lambda x : ref0 + ((x-data0)/datal ) *refl , data.prof)
            
            
            
        def draw_map(x0, x1, ymap, res):
            for n, (a, b) in enumerate(res) :
                scn.addLine(x0, ref.prof[a],x1, ymap[n] )
                scn.addLine(x0, ref.prof[b+1],x1, ymap[n+1] )

            
        def draw_boxes(data, x, ys=None):
            if ys is None : ys = data.prof
            start = data.normal
            for n in xrange(len(ys)-1) :
                y0 = ys[n]
                y1 = ys[n+1]
                
                if start : b = brushes[1]
                else : b = brushes[0]
                start = not start
                
                
                if data.desc[n]   :
                    tooltip = u'%s (%g -> %g)'%(data.desc[n], data.prof[n], data.prof[n+1])
                else :
                    tooltip = u'%g -> %g'%(data.prof[n], data.prof[n+1])
                
                scn.addRect(x, y0, 10, y1-y0, QtGui.QPen(), b).setToolTip(tooltip)
                
        draw_boxes(ref, 0)
        
        
        scn_x = -5
        scn_w = 20
        
        y = ref.prof[0]
        h = ref.prof[-1] -y
        
        by = y
        bh = h
        
        
        if data :
            hh0 = y+h
            hh1 = y
            
            if self.get_res() :
                res = self.get_res()
                y_res = map_y(res[0][0], res[-1][1]+1)
                
                
                draw_boxes(data, 30, y_res)
                draw_map(10, 30,y_res ,res)

                hh0 = y_res[0]
                hh1 = y_res[-1]
                scn_w += 30
                
            if self.get_comp() :
                comp = self.get_comp()
                #print comp
                y_comp= map_y(comp[0][0], comp[-1][1]+1)
                draw_boxes(data, -30, y_comp)
                draw_map(0, -20,y_comp ,comp)
                
                hh0 = min(hh0, y_comp[0])
                hh1 = max(hh1, y_comp[-1])
                scn_w += 30
                scn_x -= 30
                
            if hh1 >hh0 :
                by = hh0
                bh = hh1-hh0
                
        
        # best 
        by -= bh/ 100
        bh +=  bh/50
        y -= h/200.
        h += h/100.
        
        self._scene_size = (scn_x, scn_w,y, h, by, bh )
        #resize a little Scene Rect
        r= scn. sceneRect()
        r.setY(y)
        r.setHeight(h)
        scn.setSceneRect(r)
        self.zoom()
        
    def pos_label(self, x, y):
        return u'Ref:%g'%y


class SedView(BaseGView):
    def resizeEvent (self, evt ) :
        
        self.update_size()
        QtGui.QGraphicsView.resizeEvent(self, evt)
    
    #def init(self):
        
        
    def update_size(self):
        self.fitInView(self.scene().sceneRect())
        
    def redraw(self):
        scn =self._scn = QtGui.QGraphicsScene(self)
        self.setScene(self._scn)
        if not self.get_ref() or not self.get_data() or  (not self.get_comp () and not self.get_res() ) : 
            return
            
        
            
        data = self.get_data()
        ref = self.get_ref()
        
        
        def draw(d, color):
            p = QtGui.QPen(QtGui.QColor(color))
            for n, i in enumerate(d) :
                a, b = i
                scn.addLine(-ref.prof[a] , data.prof[n],-ref.prof[b+1], data.prof[n+1], p).setToolTip(color)
            
        if self.get_comp() :
            draw(self.get_comp(), 'green')
        if self.get_res() : 
            draw(self.get_res(), 'red')
        
        # grow scene
        r = QtCore.QRectF(scn.sceneRect())
        dh = r.height()*0.01
        r.setHeight(r.height()+dh)
        r.setY(r.y()-dh/2)
        dw = r.width()*.01
        r.setWidth(r.width()+dw)
        r.setX(r.x()-dw/2)
        scn.setSceneRect(r)
        self.update_size()

    def pos_label(self, x, y):
        return u'Ref:%g Data:%g'%(-x, y)

#========================================
# Main Window
#========================================


class MainWnd(QtGui.QWidget):
    def __init__(self):
        QtGui.QWidget.__init__(self, None)
        
        self.resize(800, 600)
        self.setWindowTitle(app_name)

        w = self.mouse_status = QtGui.QLabel('---', self)
        w.setFrameStyle(w.Plain|w.Box)

        tlo = QtGui.QHBoxLayout(self)
        #left part
        
        
        
        #Graphs
        sp = QtGui.QSplitter(self)
        tlo.addWidget(sp, 1)
        
        self.dtw_view = DTWView(self , sp)
        self.sed_view  =SedView(self , sp )
        sp.setStretchFactor(0, 1)
        sp.setStretchFactor(1, 2)
        
        
        # right part
        clo = QtGui.QVBoxLayout()
        tlo.addLayout(clo)
        def cmd(name, cmd, enabled=True, tooltip=None):
            def f(*p):
                #print "clk cmd", p
                return cmd()
            b = QtGui.QPushButton( name, self)
            clo.addWidget(b)
            b.clicked.connect(f)
            if not enabled : b.setEnabled(False)
            if tooltip : b.setToolTip(tooltip)
            return b
        
        
        w = self.data_status = QtGui.QLabel('No Data', self)
        w.setFrameStyle(w.Plain|w.Box)
        clo.addWidget(w)
        cmd("Load Data", self.load_file, tooltip= 'Load new data')
        
        w = self.ref_status = QtGui.QLabel('No Ref', self)
        w.setFrameStyle(w.Plain|w.Box)
        clo.addWidget(w)
        cmd("Load Ref", self.load_ref)
        cmd('Options', self.set_options)
        self.cmd_exec = cmd('Execute', self.run)
        
        w= self.res_list = QtGui.QListWidget(self)
        w.setEnabled(False)
        w.currentRowChanged.connect(self.on_res_sel)
        clo.addWidget(w, 1)
        self.cmd_comp = cmd('Compare', self.compare, enabled=False)
        self.cmd_save = cmd('Save', self.save_result, enabled=False)
        self.cmd_write1res = cmd('Write one result', self.write_1_res, enabled=False)
        self.cmd_write_res = cmd('Write all results', self.write_res, enabled=False)
        if debug :
            cmd('Test', self.test_cmd)
        
        clo.addWidget(self.mouse_status)
        
        #clo.addStretch(1)
        cmd('&Quit', self.close)
        
        
        #option
        oe = self.optdlg = OptEdit(self)
        self.opt_nbrres = OEInt(oe,"Nbr Res",  5, min=1, tooltip='Number of results')
        self.opt_maxpath = OEInt(oe, "Max Path", 0, min=0,  tooltip='Maximum number of path searched\ndefault: max(5,2*Nbr Res)')
        self.opt_del_factor = OEFloat(oe, "Gap Factor", 0.,   tooltip='Probability to add gaps')
        self.opt_max_subst = OEInt(oe, "Max Subst", 5, min=0, tooltip ="a data segment can match 1+2*n ref segments")
        self.opt_subst_dist = OEInt(oe, "Subst Dist", 5, min=1, tooltip ="max distance to check ratio (min=1)")
        
        self.ref = None
        self.cur_dir = os.getcwd() # where to search file
        self.data = None
        self.results = None
        self.ref = None
        
        self._cur_res = None
        self._comp = None
        self._cur_row = None
        self._nbr_lres = 0 #number of results in
        self._saved = [] # saved results
        
        self.update_interface()
        self.show()
        self.load_ref(defref)
        
        if debug:
            self.load_file('test3.txt')
            self.run()


    def get_cur_res(self): return self._cur_res
    def get_comp(self): return self._comp
    def get_ref(self): return self.ref
    def get_data(self): return self.data

    def redraw_graph(self):
        self.sed_view.redraw()
        self.dtw_view.redraw()
        
        
    def error(self, text):
        QtGui.QMessageBox.critical(self, app_name+u' Error', text)
        
        
    def get_infile(self, caption, fn = None):
        if not fn :
            fn = QtGui.QFileDialog.getOpenFileName(self, caption, self.cur_dir)
            if not fn : return None
            fn = unicode(fn)
        else : 
            fn = os.path.abspath(os.path.normpath(fn))
        self.cur_dir = os.path.dirname(fn)
        ifile = InFile(fn)
        
        if not ifile.ok() :
            self.error(u"%s\n(loading '%s')"%(ifile.error(), ifile.filename()))
            return None
        
        return ifile
        
    def get_outfile(self, caption):
        fn = QtGui.QFileDialog.getSaveFileName(self, caption, self.cur_dir)
        if not fn : return None
        fn = unicode(fn)
        
        self.cur_dir = os.path.dirname(fn)
        return fn 
        
    def load_file(self, filename=None):
        d =  self.get_infile("Select data", filename)
        if not d : return
        self.data = d
        self.set_results()
        self.update_interface()
        
    def load_ref(self, filename=None):
        d =  self.get_infile("Select Ref", filename)
        if not d : return
        self.ref = d
        self.set_results(force_repaint=True)
        self.update_interface()
        
    def set_options(self):
        self.optdlg.run()
        pass
        
    def save_result(self):
        dd = self.results[self._cur_row]
        num = len(self._saved)
        self._saved.append(dd)
        self.results.append(dd)
        self.res_list.addItem(self.save_res_name(num))
        pass
    
    
    
    
    def outres(self, res = None):
        if res is None : res = self.results[:self._nbr_lres]
        return OutRes( self.ref, self.data, res)
    
    def write_res(self, filename =None):
        if not filename : 
            filename = self.get_outfile("Results file name")
            if not filename : return False
            
        ores = self.outres()
        if not ores.all_write_result(filename) :
            self.error(ores.error())
            return False
        return True
        
    def write_1_res(self,num=None,  filename =None):
        if num is None : num =self._cur_row
        if not filename : 
            filename = self.get_outfile("Result file name")
            if not filename : return False
        ores = self.outres((self.results[num],) )
        if not ores.write_result(filename, 0) :
            self.error(ores.error())
            return False
        return True
        
        
    def run(self):
        QtGui.QApplication.setOverrideCursor(Qt.WaitCursor);
        self.mouse_status.setText('Computing, please wait ....')
        self.mouse_status.repaint()
        self.repaint()
        
        #get_options
        class engine(R2DTW): pass
        
        opt = engine.Options()
        
        nbrres = self.opt_nbrres.get()
        maxpath = self.opt_maxpath.get()
        if not maxpath : maxpath = max(5, 2*nbrres)
        elif maxpath < nbrres : maxpath = nbrres
        
        opt.max_path = maxpath
        
        if True : #R2DTW options
            opt.del_factor = self.opt_del_factor.get()
            opt.max_subst = self.opt_max_subst.get()
            opt.subst_dist = self.opt_subst_dist.get()

        res = engine(self.ref.todtw(), self.data.todtw(), options=opt)
        self.set_results( tuple(res.result[:nbrres]))
        self.update_interface()
        self.mouse_status.setText('')
        QtGui.QApplication.restoreOverrideCursor();

    def update_interface(self):
        can_exec = True
        if not self.ref or not self.ref.ok() :
            self.ref_status.setText('No Ref')
            self.ref_status.setToolTip('Please load a ref')
            can_exec = False
        else :
            reftext = u'Ref: %s'%os.path. basename(self.ref.filename())
            self.ref_status.setText(reftext)
            self.ref_status.setToolTip(self.ref.filename())
            
        if not self.data or not self.data.ok() :
            self.data_status.setText('No Data')
            self.data_status.setToolTip('Please load data')
            can_exec = False
        else :
            reftext = u'Data: %s'%os.path. basename(self.data.filename())
            self.data_status.setText(reftext)
            self.data_status.setToolTip(self.data.filename())
        
        if can_exec :  self.cmd_exec.setEnabled(True)
        else :  self.cmd_exec.setEnabled(False)
        
        has_results = bool(self.results)
        
        self.cmd_write_res.setEnabled(has_results)
        self.cmd_write1res.setEnabled(has_results)
        self.cmd_comp.setEnabled(has_results)
        self.cmd_save.setEnabled(has_results and (self._cur_row < self._nbr_lres))
        
    def save_res_name(self, n, outres = None):
        if not outres : outres = self.outres(())
        return outres.res_name2(self._saved[n], "S%02i "%n)

    def set_results(self, res = None, force_repaint = False):
        if not res  and not self.results : 
            if force_repaint : self.redraw_graph()
            return
            
        self.res_list.clear()
        
        if not res :
            self.results = None
            self.res_list.setEnabled(False)
            self._comp = None
            self._cur_res = None
            self._saved = []
            self._nbr_lres = 0
            self.redraw_graph()
            return
            
        self.results = list(res) + self._saved
        self._nbr_lres = len(res)
        
        ores = self.outres()
        
        self.res_list.clear()
        wl = self.res_list
        
        for i in ores.all_res_name() :
            wl.addItem(i)
            
        for n in range (len(self._saved) ):
            wl.addItem(self.save_res_name(n, ores))
            
        self.res_list.setEnabled(True)
        wl.setCurrentRow(0)
        
    def on_res_sel(self, n):
        if not self.results or n < 0 :
            return
            
        self._cur_row = n
        self._cur_res = self.results[n][1]
        self.redraw_graph()
        self.update_interface()

    def compare(self):
        if self._comp :
            self._comp = None
        else :
            n = self.res_list.currentRow()
            if n >= 0 :
                self._comp = self.results[n][1]
        self.update_interface()
        self.redraw_graph()
        
        
    def test_cmd(self):
        print "test"
        self.dtw_view.svg_output('test.svg')
        
mw = MainWnd()
mw.show()

def run():
    app.exec_()
    

if __name__ == '__main__' :
    app.exec_()
    print "Ok"
