#include <ParticleEngine.h>

#include <algorithm>


ParticleEngine::ParticleEngine():  max_particles_(100), // Default Constructor: 100 particles
	opacity_(0.5), trackers_opacity_(false), local_radius_(PARTICLES_LOCAL_RADIUS),
	width_(WINDOW_WIDTH), height_(WINDOW_HEIGHT),
	shader_("../data/shaders/particle.vert", nil, "../data/shaders/particle.frag", true)
{
#ifndef CPU_MODE
    particle_array_ = new Particle [max_particles_];
	fill_vbos();

    // Enable attributes
    glEnableVertexAttribArray(0);
    glEnableVertexAttribArray(1);
    glEnableVertexAttribArray(2);

	// Bind locations
    glBindAttribLocation(shader_.object(), 0, "in_Position");
    glBindAttribLocation(shader_.object(), 1, "in_Color");
    glBindAttribLocation(shader_.object(), 2, "in_Attribute");
	glLinkProgram(shader_.object());
#endif
}

ParticleEngine::ParticleEngine(int Max_Particles): max_particles_(Max_Particles), //Overloaded constructor
	opacity_(0.5), trackers_opacity_(false), local_radius_(PARTICLES_LOCAL_RADIUS),
	width_(WINDOW_WIDTH), height_(WINDOW_HEIGHT),
	shader_("../data/shaders/particle.vert", nil, "../data/shaders/particle.frag", true)
{
    particle_array_ = new Particle [max_particles_];

#ifndef CPU_MODE
	fill_vbos();

    // Enable attributes
    glEnableVertexAttribArray(0);
    glEnableVertexAttribArray(1);
    glEnableVertexAttribArray(2);

	// Bind locations
    glBindAttribLocation(shader_.object(), 0, "in_Position");
    glBindAttribLocation(shader_.object(), 1, "in_Color");
    glBindAttribLocation(shader_.object(), 2, "in_Attribute");
	glLinkProgram(shader_.object());
#endif
}

ParticleEngine::~ParticleEngine()
{
    delete [] particle_array_;
#ifndef CPU_MODE
	glBindBuffer(GL_ARRAY_BUFFER, 0); // Stops the use of the vbo
	// suppression des objets de la carte graphique
    glDeleteBuffers(3, vbo_);
#endif
}

void ParticleEngine::sort_particles() {
#ifndef CPU_MODE
	// Read buffer and update particles
	read_vbos();
#endif
	ParticleSorter sorter;
	std::sort(particle_array_, particle_array_+max_particles_, sorter);
#ifndef CPU_MODE
	// Refill VBOs
	fill_vbos(false);
#endif
}

void ParticleEngine::draw_particles() const {
	glEnable(GL_VERTEX_PROGRAM_POINT_SIZE);
#ifndef CPU_MODE
    // Enable attribute indexes
    glEnableVertexAttribArray(0);
    glEnableVertexAttribArray(1);
    glEnableVertexAttribArray(2);

	// Setup shader
	shader_.enable();
    shader_.uniform2f(shader_.get_uniform("WindowSize"), width_, height_);
    shader_.uniform(shader_.get_uniform("LocalRadius"), local_radius_);
    shader_.uniform(shader_.get_uniform("Opacity"), opacity_);
	shader_.uniform(shader_.get_uniform("TrackersOpacity"), ( trackers_opacity_ ? 0.0f : 1.0f ));
	glDrawArrays ( GL_POINTS, 0, max_particles_ );
	glDisableClientState(GL_VERTEX_PROGRAM_POINT_SIZE);
	shader_.disable();

    // Disable attribute indexes
	glDisableVertexAttribArray(0);
    glDisableVertexAttribArray(1);
    glDisableVertexAttribArray(2);
#else
	// Setup shader
	shader_.enable();
    shader_.uniform2f(shader_.get_uniform("WindowSize"), width_, height_);
    shader_.uniform(shader_.get_uniform("LocalRadius"), local_radius_);
    shader_.uniform(shader_.get_uniform("Opacity"), opacity_);
	shader_.uniform(shader_.get_uniform("TrackersOpacity"), ( trackers_opacity_ ? 0.0f : 1.0f ));

	// Draw particles
	GLint col = shader_.get_attribute("in_Color");
	GLint pos = shader_.get_attribute("in_Position");
	for ( long k = 0; k < max_particles_; ++k ) {
		const Particle& cur = particle_array_[k];
		shader_.attribute3f(pos, cur.get_x(),cur.get_y(),cur.get_z());
		shader_.attribute4f(col, cur.get_r(),cur.get_g(),cur.get_b(), cur.get_transparency());
		glBegin(GL_POINTS);
			glVertex3f(cur.get_x(),cur.get_y(),cur.get_z());
		glEnd();
	}

	// Disable shader
	shader_.disable();
#endif
}

bool ParticleEngine::update_particles(float* data_vector, int nb_props, int sizeu, int sizev, int sizew,
									  int velocity, bool pause,
									  float* points, float* ranges, float* colors) {
	if ( pause ) return false;

	// Iterate over the particles
	for ( long k = 0; k < max_particles_; ++k ) {
		Particle& cur = particle_array_[k];

		// Get particle coordinates
		bool reset = false;
		float index_x = ( cur.get_x() + 1.0f ) / 2.0f;
		float index_y = ( cur.get_y() + 1.0f ) / 2.0f;
		float index_z = ( cur.get_z() + 1.0f ) / 2.0f;

		// Check coordinates range
		if ( index_x >= 1.0f ) reset = true;
		if ( index_x <= 0.0f ) reset = true;
		if ( index_y >= 1.0f ) reset = true;
		if ( index_y <= 0.0f ) reset = true;
		if ( index_z >= 1.0f ) reset = true;
		if ( index_z <= 0.0f ) reset = true;

		// Check particles age
		cur.set_age( cur.get_age() + 0.1f );
		if ( cur.get_age() > cur.get_lifetime() ) {
			reset = true;
		}

		// Reset particle when out of range
		if ( reset == true ) {
			cur.init();
			continue;
		}

		// Get table index
		int index_xi = index_x * (float)( sizeu );
		int index_yi = index_y * (float)( sizev );
		int index_zi = index_z * (float)( sizew );
		if ( index_xi >= sizeu ) index_xi = sizeu - 1;
		if ( index_yi >= sizev ) index_yi = sizev - 1;
		if ( index_zi >= sizew ) index_zi = sizew - 1;
		int index_array = index_zi * sizeu * sizev + index_yi * sizeu + index_xi;

		// Color particles moving close to the tracker points
		for ( long l = 0; l < TRACKER_COUNT; ++l ) {
			if ( cur.get_x() < ( points[3*l+0] - ranges[3*l+0] ) ) continue;
			if ( cur.get_x() > ( points[3*l+0] + ranges[3*l+0] ) ) continue;
			if ( cur.get_y() < ( points[3*l+1] - ranges[3*l+1] ) ) continue;
			if ( cur.get_y() > ( points[3*l+1] + ranges[3*l+1] ) ) continue;
			if ( cur.get_z() < ( points[3*l+2] - ranges[3*l+2] ) ) continue;
			if ( cur.get_z() > ( points[3*l+2] + ranges[3*l+2] ) ) continue;
			cur.set_r(colors[3*l+0]);
			cur.set_g(colors[3*l+1]);
			cur.set_b(colors[3*l+2]);
			cur.set_transparency(100.0);
		}

		// Move particles
		float displacement_x = 0, displacement_y = 0, displacement_z = 0;
		if ( nb_props > 0 ) displacement_x = data_vector[nb_props*index_array];
		if ( nb_props > 1 ) displacement_y = data_vector[nb_props*index_array+1];
		if ( nb_props > 2 ) displacement_z = data_vector[nb_props*index_array+2];
		cur.set_x( cur.get_x() + displacement_x / (float)velocity );
		cur.set_y( cur.get_y() + displacement_y / (float)velocity );
		cur.set_z( cur.get_z() + displacement_z / (float)velocity );
	}

	return true;
}


#ifndef CPU_MODE

void ParticleEngine::read_vbos() {
	glBindBuffer(GL_ARRAY_BUFFER, vbo_[0]);
	float* array_location = (float*)glMapBuffer(GL_ARRAY_BUFFER, GL_READ_ONLY);
	for ( long k = 0; k < max_particles_; ++k ) {
		particle_array_[k].set_x(array_location[k*P_SIZE+0]);
		particle_array_[k].set_y(array_location[k*P_SIZE+1]);
		particle_array_[k].set_z(array_location[k*P_SIZE+2]);
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);
	glBindBuffer(GL_ARRAY_BUFFER, vbo_[1]);
	float* color2vbo = (float*)glMapBuffer(GL_ARRAY_BUFFER, GL_READ_ONLY);
	glGetBufferPointerv(GL_ARRAY_BUFFER, GL_BUFFER_MAP_POINTER, (void**)&color2vbo);
	for ( long k = 0; k < max_particles_; ++k ) {
		particle_array_[k].set_r(color2vbo[k*C_SIZE+0]);
		particle_array_[k].set_g(color2vbo[k*C_SIZE+1]);
		particle_array_[k].set_b(color2vbo[k*C_SIZE+2]);
		particle_array_[k].set_transparency(color2vbo[k*C_SIZE+3]);
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);
	glBindBuffer(GL_ARRAY_BUFFER, vbo_[2]);
	float* attributes2vbo = (float*)glMapBuffer(GL_ARRAY_BUFFER, GL_READ_ONLY);
	glGetBufferPointerv(GL_ARRAY_BUFFER, GL_BUFFER_MAP_POINTER, (void**)&attributes2vbo);
	for ( long k = 0; k < max_particles_; ++k ) {
		particle_array_[k].set_age(attributes2vbo[k*A_SIZE]);
		particle_array_[k].set_lifetime(attributes2vbo[k*A_SIZE+1]);
	}
	glUnmapBuffer(GL_ARRAY_BUFFER);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}

void ParticleEngine::fill_vbos(bool generate) {
    // Allocate three VBOs to our handle
    if ( generate ) glGenBuffers(3, vbo_);

	// Copy the vertex data from particle_array to the position buffer
    glBindBuffer(GL_ARRAY_BUFFER, vbo_[0]);
	vertex_vbo_size_ = max_particles_*(P_SIZE);
	float* array_location = new float[vertex_vbo_size_];
	memset(array_location, 0, vertex_vbo_size_*sizeof(float));
	for( int i = 0; i < max_particles_; ++i ) {
		array_location[i*P_SIZE+0] = particle_array_[i].get_x();
		array_location[i*P_SIZE+1] = particle_array_[i].get_y();
		array_location[i*P_SIZE+2] = particle_array_[i].get_z();
	}
	glBufferData(GL_ARRAY_BUFFER, max_particles_*(P_SIZE)*sizeof(float),
					array_location, GL_DYNAMIC_DRAW);
    glVertexAttribPointer((GLuint)0, P_SIZE, GL_FLOAT, GL_FALSE, 0, 0);
	delete [] array_location;

    // Copy the color data from colors to the color buffer
    glBindBuffer(GL_ARRAY_BUFFER, vbo_[1]);
	float* color2vbo = new float[max_particles_*C_SIZE];
	memset(color2vbo, 0, max_particles_*C_SIZE*sizeof(float));
	for( int i = 0; i < max_particles_; ++i ) {
		color2vbo[i*C_SIZE+0] = particle_array_[i].get_r();
		color2vbo[i*C_SIZE+1] = particle_array_[i].get_g();
		color2vbo[i*C_SIZE+2] = particle_array_[i].get_b();
		color2vbo[i*C_SIZE+3] = particle_array_[i].get_transparency();
	}
	glBufferData(GL_ARRAY_BUFFER, max_particles_*C_SIZE*sizeof(float),
					color2vbo, GL_DYNAMIC_DRAW);
    glVertexAttribPointer((GLuint)1, C_SIZE, GL_FLOAT, GL_FALSE, 0, 0);
	delete [] color2vbo;

    // Copy the attributes data from attributes to the attributes buffer
    glBindBuffer(GL_ARRAY_BUFFER, vbo_[2]);
	float* attributes2vbo = new float[max_particles_*A_SIZE];
	memset(attributes2vbo, 0, max_particles_*A_SIZE*sizeof(float));
	for( int i = 0; i < max_particles_; ++i ) {
		attributes2vbo[i*A_SIZE] = particle_array_[i].get_age();
		attributes2vbo[i*A_SIZE+1] = particle_array_[i].get_lifetime();
		attributes2vbo[i*A_SIZE+2] = 0.0;
	}
	glBufferData(GL_ARRAY_BUFFER, max_particles_*A_SIZE*sizeof(float),
					attributes2vbo, GL_DYNAMIC_DRAW);
    glVertexAttribPointer((GLuint)2, A_SIZE, GL_FLOAT, GL_FALSE, 0, 0);
	delete [] attributes2vbo;

	// Reset buffer
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}

GLuint ParticleEngine::get_vbo_location() const {
	 glBindBuffer(GL_ARRAY_BUFFER, vbo_[0]);
	return vbo_[0];
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}

GLuint ParticleEngine::get_vbo_color() const {
	glBindBuffer(GL_ARRAY_BUFFER, vbo_[1]);
	return vbo_[1];
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}

GLuint ParticleEngine::get_vbo_attributes() const {
	glBindBuffer(GL_ARRAY_BUFFER, vbo_[2]);
	return vbo_[2];
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}


#endif





ParticleSorter::ParticleSorter() {
	glGetFloatv(GL_MODELVIEW_MATRIX, &(matrix_[0][0]));
}

ParticleSorter::~ParticleSorter() {}

bool ParticleSorter::operator() (const Particle& rhs, const Particle& lhs) const {
	return ( z_particle(rhs) < z_particle(lhs) );
}

float ParticleSorter::z_particle(const Particle& particle) const {
	return matrix_[0][2]*particle.get_x() + matrix_[1][2]*particle.get_y() + matrix_[2][2]*particle.get_z() + matrix_[3][2];
}
