#define NO_COMPILE
#ifdef NO_COMPILE

#include <Shader.h>

#include <GL/glew.h>
#include <iostream>
#include <fstream>

#define nil 0

//! Default constructor
/*!
 @note The shaders are not initialized, so the
       GLSL program cannot be used until the
	   set_shaders() function is called
 */
GLSLShader::GLSLShader() : object_(0), vertexShader_(0), geometryShader_(0), fragmentShader_(0), ok_(false) {
	if ( !check_glew() ) return;
	object_ = glCreateProgram();
}

//! Constructor
/*!
 @param VertexShader   A string holding either the source code of the vertex
                       shader, or a path to a file which contains this source
					   code, depending on the from_file flag
 @param GeometryShader A string holding either the source code of the geometry
                       shader, or a path to a file which contains this source
					   code, depending on the from_file flag
 @param FragmentShader A string holding either the source code of the fragment
                       shader, or a path to a file which contains this source
					   code, depending on the from_file flag
 @param from_file If set to true, the parameters VertexShader, GeometryShader
                  and FragmentShader should contains a path to their respective
				  source codes. Otherwise, they should contain the source code
				  itself
 @param input_type The type of primitives accepted by the geometry shader as an input
 @param output_type The type of primitives output by the geometry shader
 @param vertices_count The maximal number of vertices output by the geometry shader
 */
GLSLShader::GLSLShader(const char* VertexShader, const char* GeometryShader, const char* FragmentShader, bool from_file,
		const GeometryShaderInputType& input_type, const GeometryShaderOutputType& output_type, int vertices_count) :
	object_(0), vertexShader_(0), geometryShader_(0), fragmentShader_(0), ok_(false)
{
	if ( !check_glew() ) return;
	object_ = glCreateProgram();
	set_shader(VertexShader, GeometryShader, FragmentShader, from_file, input_type, output_type, vertices_count);
}


//! Destructor
/*!
 @note Clean the shaders and the GLSL program
 */
GLSLShader::~GLSLShader() {
	clean_shaders();
	glDeleteProgram(object_);
}


//! Activates the shader
/*!
 @note The shader will be in use until the disable() function
       is called
 */
void GLSLShader::enable() const {
	glUseProgram(object_);
}


//! Desactivates the shader
/*!
 @note After a call of this function, all graphic primitives
       will be drawn using the OpenGL fixed-functionalities
 */
void GLSLShader::disable() const {
	glUseProgram(0);
}


//! Find the identifier of a vertex attribute
/*!
 @return The handler of the given shader attribute variable
 */
GLint GLSLShader::get_attribute(const char* Name) const {
	return glGetAttribLocation(object_, Name);
}


//! Find the identifier of a uniform attribute
/*!
 @return The handler of the given shader uniform variable
 */
GLint GLSLShader::get_uniform(const char* Name) const {
	return glGetUniformLocation(object_, Name);
}


//! Sets the value of a shader uniform variable of int type
/*!
 @param Name Identifier of the uniform variable, obtained with the get_uniform() function
 @param Value Value to set the uniform variable to
 */
void GLSLShader::uniform(GLint Name, int Value) const {
	CHECK_SHADER_ID(Name)
	glUniform1i(Name, Value);
}


//! Sets the value of a shader uniform variable of float type
/*!
 @param Name Identifier of the uniform variable, obtained with the get_uniform() function
 @param Value Value to set the uniform variable to
 */
void GLSLShader::uniform(GLint Name, float Value) const {
	CHECK_SHADER_ID(Name)
	glUniform1f(Name, Value);
}


//! Sets the value of a shader uniform variable of vec2 type
/*!
 @param Name Identifier of the uniform variable, obtained with the get_uniform() function
 @param v0 Value to set the first component of the uniform variable to
 @param v1 Value to set the second component of the uniform variable to
 */
void GLSLShader::uniform2f(GLint Name, float v0, float v1 ) const {
	CHECK_SHADER_ID(Name)
	glUniform2f(Name, v0,v1);
}


//! Sets the value of a shader uniform variable of vec3 type
/*!
 @param Name Identifier of the uniform variable, obtained with the get_uniform() function
 @param v0 Value to set the first component of the uniform variable to
 @param v1 Value to set the second component of the uniform variable to
 @param v2 Value to set the third component of the uniform variable to
 */
void GLSLShader::uniform3f(GLint Name, float v0, float v1, float v2 ) const {
	CHECK_SHADER_ID(Name)
	glUniform3f(Name, v0,v1,v2);
}


//! Sets the value of a shader uniform variable of vec4 type
/*!
 @param Name Identifier of the uniform variable, obtained with the get_uniform() function
 @param v0 Value to set the first component of the uniform variable to
 @param v1 Value to set the second component of the uniform variable to
 @param v2 Value to set the third component of the uniform variable to
 @param v3 Value to set the fourth component of the uniform variable to
 */
void GLSLShader::uniform4f(GLint Name, float v0, float v1, float v2, float v3 ) const {
	CHECK_SHADER_ID(Name)
	glUniform4f(Name, v0,v1,v2,v3);
}


//! Sets the value of a shader attribute variable of float type
/*!
 @param Name Identifier of the attribute variable, obtained with the get_attribute() function
 @param Value Value to set the attribute variable to
 */
void GLSLShader::attribute(GLint Name, float Value) const {
	CHECK_SHADER_ID(Name)
	glVertexAttrib1f(Name, Value);
}


//! Sets the value of a shader attribute variable of vec2 type
/*!
 @param Name Identifier of the attribute variable, obtained with the get_attribute() function
 @param v0 Value to set the first component of the attribute variable to
 @param v1 Value to set the second component of the attribute variable to
 */
void GLSLShader::attribute2f(GLint Name, float v0, float v1 ) const {
	CHECK_SHADER_ID(Name)
	glVertexAttrib2f(Name, v0,v1);
}


//! Sets the value of a shader attribute variable of vec3 type
/*!
 @param Name Identifier of the attribute variable, obtained with the get_attribute() function
 @param v0 Value to set the first component of the attribute variable to
 @param v1 Value to set the second component of the attribute variable to
 @param v2 Value to set the third component of the attribute variable to
 */
void GLSLShader::attribute3f(GLint Name, float v0, float v1, float v2 ) const {
	CHECK_SHADER_ID(Name)
	glVertexAttrib3f(Name, v0,v1,v2);
}


//! Sets the value of a shader attribute variable of vec4 type
/*!
 @param Name Identifier of the attribute variable, obtained with the get_attribute() function
 @param v0 Value to set the first component of the attribute variable to
 @param v1 Value to set the second component of the attribute variable to
 @param v2 Value to set the third component of the attribute variable to
 @param v3 Value to set the fourth component of the attribute variable to
 */
void GLSLShader::attribute4f(GLint Name, float v0, float v1, float v2, float v3 ) const {
	CHECK_SHADER_ID(Name)
	glVertexAttrib4f(Name, v0,v1,v2,v3);
}


//! Get the status of the shader
/*!
 @return True if shader is valid and ready to use, false otherwise
 */
bool GLSLShader::ok() const {
	return ok_;
}


//! Sets the shader source code, compiles and links
/*!
 @param VertexShader   A string holding either the source code of the vertex
                       shader, or a path to a file which contains this source
					   code, depending on the from_file flag
 @param GeometryShader A string holding either the source code of the geometry
                       shader, or a path to a file which contains this source
					   code, depending on the from_file flag
 @param FragmentShader A string holding either the source code of the fragment
                       shader, or a path to a file which contains this source
					   code, depending on the from_file flag
 @param from_file If set to true, the parameters VertexShader, GeometryShader
                  and FragmentShader should contains a path to their respective
				  source codes. Otherwise, they should contain the source code
				  itself
 @param input_type The type of primitives accepted by the geometry shader as an input
 @param output_type The type of primitives output by the geometry shader
 @param vertices_count The maximal number of vertices output by the geometry shader
 */
void GLSLShader::set_shader(const char* VertexShader, const char* GeometryShader, const char* FragmentShader, bool from_file,
		const GeometryShaderInputType& input_type, const GeometryShaderOutputType& output_type, int vertices_count) {
	// Clean previous shaders, if any
	clean_shaders();
	if ( glCreateShader == NULL ) glewInit();

    // Vertex shader compiling
    if(VertexShader) {
		const char* VertexSource = nil;
		std::string VertexSourceStr;

		// Get vertex shader source code
		if ( from_file ) {
			VertexSourceStr = load_file(VertexShader);
			VertexSource = VertexSourceStr.c_str();
		}
		else VertexSource = VertexShader;

		// Create and compile the shader
		vertexShader_ = glCreateShader(GL_VERTEX_SHADER);
		glShaderSource(vertexShader_, 1, &VertexSource, NULL);
		glCompileShader(vertexShader_);

		// Check compilation
		if(!check_compiling(vertexShader_, "Vertex Shader error - cannot compile.\n")) {
			ok_ = false;
			return;
		}

		// Attach the shader
		glAttachShader(object_, vertexShader_);
	}

    // Geometry shader compiling
    if(GeometryShader) {
		std::string GeometrySourceStr;
		const char* GeometrySource = nil;

		// Get geometry shader source code
		if ( from_file ) {
			GeometrySourceStr = load_file(GeometryShader);
			GeometrySource = GeometrySourceStr.c_str();
		}
		else GeometrySource = GeometryShader;

		// Create and compile the shader
		geometryShader_ = glCreateShader(GL_GEOMETRY_SHADER_EXT);
		glShaderSource(geometryShader_, 1, &GeometrySource, NULL);
		glCompileShader(geometryShader_);

		// Check compilation
		if(!check_compiling(geometryShader_, "Geometry Shader error - cannot compile.\n")) {
			ok_ = false;
			return;
		}

		// Attach the shader
		glAttachShader(object_, geometryShader_);

		// Set the shader parameters
		glProgramParameteriEXT(object_,GL_GEOMETRY_INPUT_TYPE_EXT,input_type);
		glProgramParameteriEXT(object_,GL_GEOMETRY_OUTPUT_TYPE_EXT,output_type);
		glProgramParameteriEXT(object_,GL_GEOMETRY_VERTICES_OUT_EXT,vertices_count);
	}

	// Fragment shader compiling
    if(FragmentShader) {
		std::string FragmentSourceStr;
		const char* FragmentSource = nil;

		// Get fragment shader source code
		if ( from_file ) {
			FragmentSourceStr = load_file(FragmentShader);
			FragmentSource = FragmentSourceStr.c_str();
		}
		else FragmentSource = FragmentShader;

		// Create and compile the shader
		fragmentShader_ = glCreateShader(GL_FRAGMENT_SHADER);
		glShaderSource(fragmentShader_, 1, &FragmentSource, NULL);
		glCompileShader(fragmentShader_);

		// Check compilation
		if(!check_compiling(fragmentShader_, "Fragment Shader error - cannot compile.\n")) {
			ok_ = false;
			return;
		}

		// Attach the shader
		glAttachShader(object_, fragmentShader_);
    }

    // Link the program
	glLinkProgram(object_);

	// Check the link
	if ( !check_linking(object_, "Shader error - cannot link shader.\n") ) ok_ = false;
	else ok_ = true;
}


//! Check whether the shader was correctly compiled
/*!
 @param Shader The identifier of the shader to be checked
 @param Message The message to be output in case of error
 @return True if compilation was successful, false otherwise
 */
bool GLSLShader::check_compiling(int Shader, const char* Message) const {
	// Check compile status
    int Success;
    glGetShaderiv(Shader, GL_COMPILE_STATUS, &Success);

	// Print error message in case of failure
	if(Success == GL_FALSE) {
		// Print custom error message
        std::cout << Message << std::endl;

		// Get error log
        int InfoLogSize;
        glGetShaderiv(Shader, GL_INFO_LOG_LENGTH, &InfoLogSize);
        char* Buffer = new char[InfoLogSize];
		glGetShaderInfoLog(Shader, InfoLogSize, NULL, Buffer);

		// Print error log
        std::cout << Buffer << std::endl;
        delete[] Buffer;

		// Return failure
		return false;
	}
    return true;
}


//! Check whether the shader was correctly linked
/*!
 @param Program The identifier of the GLSL program to be checked
 @param Message The message to be output in case of error
 @return True if link was successful, false otherwise
 */
bool GLSLShader::check_linking(int Program, const char* Message) const {
	// Check link status
    int Success;
    glGetProgramiv(Program, GL_LINK_STATUS, &Success);

	// Print error message in case of failure
	if(Success == GL_FALSE) {
		// Print custom error message
        std::cout << Message << std::endl;

		// Get error log
        int InfoLogSize;
        glGetProgramiv(Program, GL_INFO_LOG_LENGTH, &InfoLogSize);
        char* Buffer = new char[InfoLogSize];
		glGetProgramInfoLog(Program, InfoLogSize, NULL, Buffer);

		// Print error log
        std::cout << Buffer << std::endl;
        delete[] Buffer;

		// Return failure
		return false;
	}

	// Return success
    return true;
}


//! Loads a shader source code file from disk
/*!
 @param Filename The path to the file that should be read
 @return A string containing the content of the file
 */
std::string GLSLShader::load_file(const char* Filename) {
	// Get the file
	std::ifstream stream(Filename, std::ios::in);

	// Check if we were able to open the file
    if(!stream.is_open())
        return "";

	// Read the file
    std::string Line = "";
	std::string Text = "";
	while(getline(stream, Line))
    	Text += "\n" + Line;

	// Close the file
	stream.close();

	// Return the content of the file
	return Text;
}


//! Check if the glew library is correctly initialized
/*!
 @return False if glew cannot be initialized, true otherwise
 */
bool GLSLShader::check_glew() {
	// Check the create program function. If it is set to nil,
	// glew is either not yet initialized or unable to run shaders
	if ( glCreateProgram == NULL ) {
		// Try to initialize glew
		glewInit();

		// Check compatibility
		if ( GLEW_ARB_vertex_shader && GLEW_ARB_fragment_shader && GLEW_GET_VAR(__GLEW_EXT_geometry_shader4) /* && GL_EXT_geometry_shader4 */ ) {
            std::cout << "GPU compatible with GLSL - vertex, fragment, and geometry units." << std::endl;
		}
		else if ( GLEW_ARB_vertex_shader && GLEW_ARB_fragment_shader ) {
            std::cout << "GPU compatible with GLSL - vertex and fragment units." << std::endl;
            std::cout << "GPU not compatible with GLSL - geometry unit." << std::endl;
		}
		else {
            std::cout << "GPU cannot run shaders." << std::endl;
		}

		// Check glew initialization
		if( glCreateProgram == NULL ) {
            std::cout << "Cannot initialize glew." << std::endl;
			return false;
		}
	}

	return true;
}


//! Deletes the shader
void GLSLShader::clean_shaders() {
	if ( vertexShader_ ) {
		if ( object_ ) glDetachShader(object_, vertexShader_);
		glDeleteShader(vertexShader_);
		vertexShader_ = 0;
	}

	if ( geometryShader_ ) {
		if ( object_ ) glDetachShader(object_, geometryShader_);
		glDeleteShader(geometryShader_);
		geometryShader_ = 0;
	}

	if ( fragmentShader_ ) {
		if ( object_ ) glDetachShader(object_, fragmentShader_);
		glDeleteShader(fragmentShader_);
		fragmentShader_ = 0;
	}

	ok_ = false;
}

#endif
