#ifndef shader_h
#define shader_h

#include <OpenGL.h>

#include <string>

#define VERBOSE_SHADERS

//! Check whether a shader entry is valid (>0) and display an error message if it is not
/*!
 @note For the sake of efficiency, this only works in debug mode
 */
#ifndef _DEBUG
	#define CHECK_SHADER_ID(ID) {}
#else
	#ifdef VERBOSE_SHADERS
		#define CHECK_SHADER_ID(ID) { \
			if ( ID < 0 ) \
				std::cout << "Error at line " << __LINE__ << " in file " << __FILE__ << " - \"" << ID << "\" is not a valid shader ID." << std::endl;\
			}
	#else
		#define CHECK_SHADER_ID(ID) {}
	#endif
#endif

typedef unsigned int GLuint;
typedef int GLint;


/*!
 *  Class in charge of holding shaders written in GLSL
 *
 *  Shaders are small pieces of code executed directly on
 *  the graphic processing unit (GPU). They allow a great
 *  flexibility in the treatment of graphic primitives,
 *  using a SIMD (Single Instruction Multiple Data)
 *  architecture. The shaders can be compiled and linked at
 *  run-time, allowing to change dynamically their content
 *  while the main application is running.
 *
 *  GLSL stands for the "OpenGL Shading Language". Its syntax
 *  is close to the C language. An excellent overview of the
 *  GLSL language can be found in Randi Rost's book, "OpenGL
 *  Shading Language, second edition", aka the "Orange book"
 */
class GLSLShader {

public:
	enum GeometryShaderInputType {
		PointsIn = GL_POINTS,
		LinesIn = GL_LINES,
		LinesAdjacencyIn = GL_LINES_ADJACENCY_EXT,
		TrianglesIn = GL_TRIANGLES,
		TrianglesAdjacencyIn = GL_TRIANGLES_ADJACENCY_EXT
	};
	enum GeometryShaderOutputType {
		PointsOut = GL_POINTS,
		LineStripOut = GL_LINE_STRIP,
		TriangleStripOut = GL_TRIANGLE_STRIP
	};
	GLSLShader();
	GLSLShader(const char* VertexShader, const char* GeometryShader, const char* FragmentShader, bool from_file=true,
		const GeometryShaderInputType& input_type = PointsIn, const GeometryShaderOutputType& output_type = TriangleStripOut,
		int vertices_count = 12);
	~GLSLShader();

	void set_shader(const char* VertexShader, const char* GeometryShader, const char* FragmentShader, bool from_file=true,
		const GeometryShaderInputType& input_type = PointsIn, const GeometryShaderOutputType& output_type = TriangleStripOut,
		int vertices_count = 12);

	void enable() const;
	void disable() const;

	GLint get_attribute(const char* Name) const;
	GLint get_uniform(const char* Name) const;

	void uniform(GLint Name, int Value) const;
	void uniform(GLint Name, float Value) const;
	void uniform2f(GLint Name, float, float) const;
	void uniform3f(GLint Name, float, float, float) const;
	void uniform4f(GLint Name, float,float,float,float) const;

	void attribute(GLint Name, float Value) const;
	void attribute2f(GLint Name, float, float) const;
	void attribute3f(GLint Name, float, float, float) const;
	void attribute4f(GLint Name, float,float,float,float) const;

	bool ok() const;

	static bool check_glew();

	GLuint object() { return object_; }


private:
	std::string load_file(const char* Filename);
	bool check_compiling(int Shader, const char* Message) const;
	bool check_linking(int Program, const char* Message) const;
	void clean_shaders();

	//! Identifier of the GLSL program
	GLuint object_;

	//! Identifier of the vertex shader
	GLuint vertexShader_;
	//! Identifier of the geometry shader
	GLuint geometryShader_;
	//! Identifier of the fragment shader
	GLuint fragmentShader_;

	//! State of the shader
	bool ok_;
};

#endif// UncertaintyViewer_gtools_shader_h

