//! Load a 2D grid stored in the GSlib format
template <class T>
T* GSlib::load_file(int& nb_props, int& size_u, int& size_v,
					std::list<std::string>& property_names, const char* filename) {
	// Access raw file
	std::ifstream in(filename);
	if ( !in.good() ) {
		std::cout << "Input file error" << std::endl;
		return nil;
	}

	// Get texture size
	{{
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		if ( list.size() < 2 ) {
			std::cout << "Cannot read U and V sizes. First line of input file must be \"U V\", where U and V are integers." << std::endl;
			return nil;
		}
		std::list<std::string>::const_iterator it = list.begin();
		size_u = string_convert<int>(*it);
		++it;
		size_v = string_convert<int>(*it);
	}}

	// Get number of properties
	{{
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		std::list<std::string>::const_iterator it = list.begin();
		nb_props = string_convert<int>(*it);
	}}

	// Get property names
	for ( int k = 0; k < nb_props; ++k ) {
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		std::list<std::string>::const_iterator it = list.begin();
		property_names.push_back(*it);
	}

	// Generate float array
	T* res = new T[nb_props*size_u*size_v];

	// Iterate over the texels
	long index = 0;
	for ( int u = 0; u < size_u; ++u ) {
		for ( int v = 0; v < size_v; ++v ) {
			// Get line
			char lineStr[1024];
			in.getline(lineStr, 1024);
			std::string line(lineStr);

			// Split line
			std::list<std::string> list;
			string_split(list, line);

			// Get texel value
			std::list<std::string>::const_iterator it = list.begin();
			int i = 0;
			while ( ( i < nb_props ) && ( it != list.end() ) ) {
				res[index+i] = string_convert<T>(*it);
				++i; ++it;
			}
			index += nb_props;
		}
	}

	// Return result
	return res;
}

//----------------------------------------

//! Load a 3D grid stored in the GSlib format
template <class T>
T* GSlib::load_file(int& nb_props, int& size_u, int& size_v, int& size_w,
					std::list<std::string>& property_names, const char* filename) {
	// Access raw file
	std::ifstream in(filename);
	if ( !in.good() ) {
		std::cout << "Input file error" << std::endl;
		return nil;
	}

	// Get texture size
	{{
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		if ( list.size() < 3 ) {
			std::cout << "Cannot read (U,V,W) sizes. First line of input file must be \"U V W\", where U, V and W are integers." << std::endl;
			return nil;
		}
		std::list<std::string>::const_iterator it = list.begin();
		size_u = string_convert<int>(*it);
		++it;
		size_v = string_convert<int>(*it);
		++it;
		size_w = string_convert<int>(*it);
	}}

	// Get number of properties
	{{
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		std::list<std::string>::const_iterator it = list.begin();
		nb_props = string_convert<int>(*it);
	}}

	// Get property names
	for ( int k = 0; k < nb_props; ++k ) {
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		std::list<std::string>::const_iterator it = list.begin();
		property_names.push_back(*it);
	}

	// Generate float array
	T* res = new T[nb_props*size_u*size_v*size_w];

	// Iterate over the texels
	// MODIFICATIONS
	for ( int u = 0; u < size_u; ++u ) {	
		for ( int v = 0; v < size_v; ++v ) {
			for ( int w = 0; w < size_w; ++w ) {
				long index = nb_props*(u + v*size_u + w*size_u*size_v);
				// Get line
				char lineStr[1024];
				in.getline(lineStr, 1024);
				std::string line(lineStr);
//				std::cout << "Cell " << index/nb_props << " = \"" << line << "\"" << std::endl;

				// Split line
				std::list<std::string> list;
				string_split(list, line);

				// Get texel value
				std::list<std::string>::const_iterator it = list.begin();
				int i = 0;
				while ( ( i < nb_props ) && ( it != list.end() ) ) {
					res[index+i] = string_convert<T>(*it);
					++i; ++it;
				}
			}
		}
	}

	// Return result
	return res;
}

//----------------------------------------

//! Load a raw file stored in the GSlib format
template <class T>
T* GSlib::load_raw_file( int& size, const char* filename) {
	// Access raw file
	std::ifstream in(filename);
	if ( !in.good() ) {
		std::cout << "Input file error" << std::endl;
		return nil;
	}

	// Get texture size
	{{
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		if ( list.size() < 1 ) {
			std::cout << "Cannot read raw file size. First line of input file must be \"size [...]\", where size is an integer." << std::endl;
			return nil;
		}
		std::list<std::string>::const_iterator it = list.begin();
		size = string_convert<int>(*it);
	}}

	// Generate float array
	T* res = new T[size];

	// Iterate over the texels
	long index = 0;
	for ( int k = 0; k < size; ++k ) {
		// Get line
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);

		// Split line
		std::list<std::string> list;
		string_split(list, line);

		// Get texel value
		std::list<std::string>::const_iterator it = list.begin();
		res[index] = string_convert<T>(*it);
		++index;
	}

	// Return result
	return res;
}

//----------------------------------------

//! Split a string into a set of substrings
void GSlib::string_split(std::list<std::string>& result, const std::string& str) {
	result.clear();
	std::istringstream iss(str);
	while (iss) {
		std::string sub;
		iss >> sub;
		if ( !sub.empty() ) result.push_back(sub);
	}
}

//----------------------------------------

//! Convert a string into any numeric type
template <class T>
T GSlib::string_convert(const std::string& str) {
	T result;
	std::istringstream iss(str);
	iss >> result;
	return result;
}
