// Vertex shader

//----------------------------------------

#define SIZE_LIMIT 0.6

//----------------------------------------

// Size of the screen
uniform vec2 WindowSize;

// Local radius defined by the user
uniform float LocalRadius;

// Scale data
varying vec2 coeffs;
varying float radius;

// Particle attributes
attribute vec3 in_Position;
attribute vec4 in_Color;
attribute vec3 in_Attribute;
varying vec4 ex_Color;

// Center of the point (pixel coordinates)
varying vec2 pos;

//----------------------------------------

vec2 zoom_factor(vec3 world_position);

//----------------------------------------

void main() {
	// Output color
	ex_Color = in_Color;

	// Set the radius and send it to the fragment shader
	vec2 PointScales = zoom_factor(in_Position);
	if ( PointScales.x < SIZE_LIMIT ) PointScales = vec2(SIZE_LIMIT, PointScales.y*SIZE_LIMIT/PointScales.x);
	if ( PointScales.y < SIZE_LIMIT ) PointScales = vec2(PointScales.x*SIZE_LIMIT/PointScales.y, SIZE_LIMIT);
	float radius_tmp = LocalRadius / max(PointScales.x,PointScales.y);

	// Change the size of the point to take into account the zoom level
	coeffs = vec2( min(PointScales.x / PointScales.y, 1.0), min(PointScales.y / PointScales.x, 1.0) );
	gl_PointSize = 2.0 * radius_tmp / min(coeffs.x, coeffs.y);
	radius = radius_tmp * radius_tmp;

	// Convert position of the point to window coordinates
	vec4 position = gl_ModelViewProjectionMatrix * vec4(in_Position,1);
	pos = ( vec2(1.0) + (position.xy / position.w)) * WindowSize * 0.5;

	// Transform the vertex
	gl_Position = position;
}


//----------------------------------------

vec2 zoom_factor(vec3 world_position) {
	// Change coordinate system
	vec4 modelCoordinates = vec4(world_position,1);
	vec4 view_ori = gl_ModelViewMatrix*modelCoordinates;
	vec4 projection_ori = gl_ProjectionMatrix*view_ori;
	projection_ori /= projection_ori.w;

	// Offset points
	vec4 projection_u = projection_ori + vec4(1.0/(float)WindowSize.x,0,0,0);
	vec4 projection_v = projection_ori + vec4(0,1.0/(float)WindowSize.y,0,0);
	vec4 view_u = gl_ProjectionMatrixInverse*projection_u;
	vec4 view_v = gl_ProjectionMatrixInverse*projection_v;

	// Return result
	return vec2(length(view_ori.xyz-view_u.xyz), length(view_ori.xyz-view_v.xyz));
}
