#include <iostream>
#include <iomanip>
#include <vector>
#include <map>
#include <sstream>
#include <strstream>
#include <list>
#if defined(_WIN32)
	#include <windows.h>
	#ifndef i64
		typedef __int64 i64;
	#endif
#endif

using namespace std;

#include <GL/glew.h>
#include <GL/glut.h>
#include <NV/nvGlutManipulators.h>

#include <ParticleEngine.h>
#include <utils.h>
#include <Define.h>
#include <MersenneTwister.h>
#include <gslib.h>

// Text display list
GLuint textDList = 0;

// Manipulator
nv::GlutExamine manipulator;

// Initial size of the window
int width = WINDOW_WIDTH;
int height = WINDOW_HEIGHT;

// Global state
bool draw_cube = true;
bool draw_axis = true;
bool z_buffer = false;
bool draw = true;

// GSLIB variables
int nb_props, sizeu, sizev, sizew;

// Particle counts
int pcount[] = {
	500, 5000, 50000, 100000, 500000, 1000000
};
int pindex = 2;

// Create Particle Engine
ParticleEngine* engine;// = nil;
#ifndef CPU_MODE
	GLuint vbo_location;
	GLuint vbo_color;
	GLuint vbo_attributes;
	GLuint vbo_vector[1];
	GLuint vbo_vector_cuda;
#endif

int velocity = 100;
bool pause = false;

float* points = nil;
float* ranges = nil;
float* colors = nil;

// Timer data
#if defined(_WIN32)
	i64 clocks_;
	i64 freq_;
	float fps_ = 0.0f;
	int frames_ = 0;
#endif

#ifndef CPU_MODE
	extern "C" void initCuda();
	extern "C" void resetCuda(float* points, float* ranges, float* colors, int size_trackers);
	extern "C" void runCuda(GLuint vbo_location, GLuint vbo_color, GLuint vbo_attributes,
							GLuint vbo_vector_cuda, int nb_props, int sizeu, int sizev, int sizew,
							int velocity, bool pause, int pcount);
	extern "C" void closeCuda();
#endif

float* data_vector = nil;
int size_vector = 0;
int gslib = 3;
std::string gslib_file = "";

//----------------------------------------

void load_gslib();
void reset_engine();
#ifndef CPU_MODE
	void fill_vbo_vector(float* data_vector, int size_vector, int nb_props);
#endif
void init_trackers();

//----------------------------------------

#ifndef CPU_MODE
	GLuint get_vbo_vector_cuda(){
		glBindBuffer(GL_ARRAY_BUFFER, vbo_vector[0]);
		return vbo_vector[0];
		glBindBuffer(GL_ARRAY_BUFFER, 0);
	}
#endif

//----------------------------------------

// Initialize OpenGL state
void init_opengl() {
	// Init GLEW
	glewInit();

	// Query clock data
#if defined(_WIN32)
	QueryPerformanceCounter((LARGE_INTEGER *)&clocks_);
	QueryPerformanceFrequency((LARGE_INTEGER *)&freq_);
#endif

	// Check GPU supported features
	if (!glewIsSupported( "GL_VERSION_2_0 GL_ARB_fragment_program" )) {
		printf( "Error: failed to get minimal extensions for demo\n");
		printf( "This sample requires:\n");
		printf( "  OpenGL version 2.0\n");
		printf( "  GL_ARB_fragment_program\n");
		exit(-1);
	}

	// Setup default OpenGL state
	if (z_buffer == false) glDisable(GL_DEPTH_TEST);
	else glEnable(GL_DEPTH_TEST);
	glEnable(GL_BLEND);
	glBlendFunc(GL_SRC_ALPHA,GL_ONE_MINUS_SRC_ALPHA);
	glClearColor(1.0, 1.0, 1.0, 1.0);
	glDepthFunc(GL_LEQUAL);

	// Generate display lists for ASCII characters
	textDList = glGenLists(128);
	for ( int k = 0; k < 128; ++k) {
		glNewList( k + textDList, GL_COMPILE);
		glutBitmapCharacter(  GLUT_BITMAP_HELVETICA_12, k );
		glEndList();
	}
	glListBase(textDList);
}

//----------------------------------------

void display_text(const char* text) {
	// Setup OpenGL state
	glMatrixMode(GL_PROJECTION);
	glPushMatrix();
	glLoadIdentity();
	glOrtho( 0.0, double(width), 0.0, double(height), 0.0, 1.0);
	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glLoadIdentity();

	// Render a text overlay
	char size[256];
	glColor3f(1.0-BACKGROUND_COLOR_R,1.0-BACKGROUND_COLOR_G,1.0-BACKGROUND_COLOR_B);
	sprintf( size, text);
	glRasterPos2f( 10.0f, height - 20.0f);
	glCallLists( (GLsizei)strlen(size), GL_UNSIGNED_BYTE, size);
	glBindTexture( GL_TEXTURE_2D, 0);

	// Reset matrices
	glMatrixMode(GL_PROJECTION);
	glPopMatrix();
	glMatrixMode(GL_MODELVIEW);
	glPopMatrix();
}

//----------------------------------------

// Display data
void display() {
	// Clear previous display
	{{
		glClearColor(BACKGROUND_COLOR_R,BACKGROUND_COLOR_G,BACKGROUND_COLOR_B,0);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	}}

	// Draw the data
	{{
		if ( draw ) {
			// Setup OpenGL state
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();

			// Apply manipulator
			manipulator.applyTransform();

			// Draw particles
			engine->draw_particles();

			// Draw cube
			if ( draw_cube == true ) {
				// Borders
				drawCube(1.0, 0.5);

				// Trackers
				for ( int k = 0; k < TRACKER_COUNT; ++k ) {
					drawCube(1.0, points[3*k], points[3*k+1], points[3*k+2],
									ranges[3*k], ranges[3*k+1], ranges[3*k+2],
									colors[3*k], colors[3*k+1], colors[3*k+2]);
				}
			}
			if ( draw_axis == true ) drawAxis(1.0, 1.0);

			// Reset OpenGL state
			glPopMatrix();
		}

		// Update particles
#ifndef CPU_MODE
		runCuda(vbo_location, vbo_color, vbo_attributes, vbo_vector_cuda, nb_props, sizeu, sizev, sizew, velocity, pause, engine->get_particle_count());
#else
		engine->update_particles(data_vector, nb_props, sizeu, sizev, sizew, velocity, pause, points, ranges, colors);
#endif
	}}

	// Get FPS
#if defined(_WIN32)
	{{
		++frames_;
		i64 clocks;
		QueryPerformanceCounter((LARGE_INTEGER *)&clocks);
		double time = ((double)(clocks-clocks_) / (double)freq_);
		if ( time >= FPS_TIME ) {
			if ( frames_ >= FPS_MIN_FRAMES ) {
				fps_ = (double)frames_ / time;
				clocks_ = clocks;
				frames_ = 0;
				if ( WINDOW_MODE == GLUT_SINGLE ) std::cout << fps_ << std::endl;
			}
		}
		if ( WINDOW_MODE != GLUT_SINGLE ) {
			std::strstream stream;
#ifdef CPU_MODE
			stream << "CPU mode - " << engine->get_particle_count() << " particles - " << gslib_file << ".gslib (";
#else
			stream << "GPU mode - " << engine->get_particle_count() << " particles - " << gslib_file << ".gslib (";
#endif
			stream << std::setprecision(2) << std::fixed << fps_ << " fps)";
			display_text(stream.str());
		}
	}}
#endif

	if ( WINDOW_MODE != GLUT_SINGLE ) glutSwapBuffers();
	else glFlush();
}

//----------------------------------------

// Reset the display when idling
void idle() {
	glutPostRedisplay();
}

//----------------------------------------

// Clean state
void cleanup() {
	delete engine;
#ifndef CPU_MODE
	std::cout << std::endl;
	std::cout<< " Closing cuda " << std::endl;
	closeCuda();
#else
	if ( data_vector != nil ) delete [] data_vector;
#endif
	if ( points != nil ) delete [] points;
	if ( ranges != nil ) delete [] ranges;
	if ( colors != nil ) delete [] colors;
}

//----------------------------------------

void help() {
	// Clear previous display
	system("cls");

	// Help on commands
	printf( "\n Particle engine commands\n\n");
	printf( "  ---------------------------------------------\n");
	printf( "     q/[ESC]    - Quit the application\n");
	printf( "  ---------------------------------------------\n");
	printf( "        p       - Play/pause \n");
	printf( "       R/r      - Increase/decrease particle radius \n");
	printf( "       V/v      - Increase/decrease particle velocity \n");
	printf( "       O/o      - Increase/decrease particle opacity \n");
	printf( "        n       - Change particle count \n");
	printf( "        b       - Enable/disable the Z-buffer \n");
	printf( "        f       - Change vector field (.gslib file) \n");
	printf( "        s       - Sort particles \n");
	printf( "        t       - Toggle colored particles highlighting \n");
	printf( "        i       - Reset colored trackers \n");
#ifdef BENCHMARKS
	printf( "        z       - Toggle drawing for particle update benchmarks \n");
#endif
//	printf( "        a       - Display/hide the axis \n");
//	printf( "        c       - Display/hide the cube \n");
	printf( "  ---------------------------------------------\n");
	printf( " [MOUSE LEFT]   - Rotate the view point \n");
	printf( " [MOUSE RIGHT]  - Zoom \n");
	printf( " [MOUSE MIDDLE] - Move the view point \n");
	printf( "  ---------------------------------------------\n");
	printf( "        X axis  - Red axis \n");
	printf( "        Y axis  - Green axis \n");
	printf( "        Z axis  - Blue axis \n");
	printf( "  ---------------------------------------------\n\n");
}

//----------------------------------------

// Function called when a key is pressed
void key(unsigned char k, int x, int y) {
	// Low character
	unsigned char lk = tolower(k);

	// Change gslib file
	if (lk=='f') {
		load_gslib();
	}
	
	// Pause/play the animation
	if (lk=='p') {
		pause = !pause;
	}
	// Sort particles
	if (lk=='s') {
		if ( engine ) {
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();
			glLoadIdentity();
			manipulator.applyTransform();
			engine->sort_particles();
			glPopMatrix();
		}
	}

	// Increase/decrease velocity
	if (k=='V') {
		if ( velocity > 5 ) velocity -= 5;
		else velocity = 1;
	}
	else if (k=='v') {
		velocity += 5;
	}

	// Increase/decrease radius
	if (k=='R') {
		float tmp_local_radius = engine->get_engine_local_radius();
        tmp_local_radius += 1;
        engine->set_engine_local_radius (tmp_local_radius) ;
	}
	else if (k=='r') {
		float tmp_local_radius = engine->get_engine_local_radius();
        if ( tmp_local_radius > 2){
               tmp_local_radius -= 1;
               engine->set_engine_local_radius (tmp_local_radius) ;
        }
	}

	// Increase/decrease opacity
	if (k=='O') {
		float tmp_opacity = engine->get_engine_opacity();
        tmp_opacity += 0.025;
		if ( tmp_opacity > 1.0 ) tmp_opacity = 1.0;
        engine->set_engine_opacity(tmp_opacity) ;
	}
	else if (k=='o') {
		float tmp_opacity = engine->get_engine_opacity();
		tmp_opacity -= 0.025;
		if ( tmp_opacity < 0.025 ) tmp_opacity = 0.025;
		engine->set_engine_opacity(tmp_opacity) ;
	}

	// Trackers opacity
	if (lk=='t') {
		bool tmp_opacity = !engine->get_engine_trackers_opacity();
		engine->set_engine_trackers_opacity(tmp_opacity) ;
	}

	// Change particle number
	if (lk=='n') {
		reset_engine();
	}

	// Reset colored trackers
	if (lk=='i') {
		init_trackers();
	}

	// Draws the cube
	if (lk=='c') {
        if ( draw_cube == true ) draw_cube = false;
        else draw_cube = true;
	}

	// Draws axis
	if (lk=='a') {
        if ( draw_axis == true ) draw_axis = false;
        else draw_axis = true;
	}

	// Enable Z-Buffer
	if (lk=='b') {
        if (z_buffer == false){
            z_buffer = true;
            glEnable(GL_DEPTH_TEST);  // Z-Buffer Initialisation
        }
        else {
            z_buffer = false;
            glDisable(GL_DEPTH_TEST);  // Z-Buffer Initialisation
        }
	}

#ifdef BENCHMARKS
	// Toggle draw
	if (lk=='z') {
		draw = !draw;
	}
#endif

	// Deal with quit key
	if(lk==27 || lk=='q') {
		cleanup();
		exit(0);
	}

	// Reset display
	glutPostRedisplay();
}

//----------------------------------------

// Resize the window
void resize(int w, int h) {
	// Check height
	if (h == 0) h = 1;

	// Update view parameters
	glViewport(0, 0, w, h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(60.0, (GLfloat)w/(GLfloat)h, 0.1, 100.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// Update manipulator
	manipulator.reshape(w, h);
	width = w;
	height = h;
	engine->set_engine_width_and_height ( width, height );
}

//----------------------------------------

// Function called when the mouse is pressed
void mouse(int button, int state, int x, int y) {
	manipulator.mouse(button, state, x, y);
}

//----------------------------------------

// Motion function
void motion(int x, int y) {
	manipulator.motion(x, y);
}

//----------------------------------------


// Main function
int main(int argc, char **argv)
{
	// Initialize GLUT and create window
	glutInit(&argc, argv);
	glutInitWindowSize( width, height);
	glutInitDisplayMode(WINDOW_MODE | GLUT_DEPTH | GLUT_RGB);
	glutCreateWindow("Particle engine");

	// Initialize OpenGL
	init_opengl();

	// Initialize colored trackers
#ifndef CPU_MODE
	std::cout << std::endl << " Storing previous information on the GPU (initCuda function) " << std::endl;
	initCuda();
#endif
	init_trackers();
	
	// Setup GLUT important functions
	glutDisplayFunc(display);
	glutMouseFunc(mouse);
	glutMotionFunc(motion);
	glutIdleFunc(idle);
	glutKeyboardFunc(key);
	glutReshapeFunc(resize);

	// Setup manipulator
	manipulator.setDollyActivate( GLUT_RIGHT_BUTTON );
	manipulator.setPanActivate( GLUT_MIDDLE_BUTTON );
	manipulator.setDollyPosition( -3.0f );

	// Display help on commands
	help();

	// Create particle engine
	reset_engine();

	// Reading GSLIB data
	load_gslib();

#ifndef CPU_MODE
	// Initialize and fill vbo_vector
	std::cout << std::endl << " Initializing and linking vbo_location " << std::endl;
	vbo_location = engine->get_vbo_location();

	// Initialize and fill vbo_vector
	std::cout << std::endl << " Initializing and linking vbo_color " << std::endl;
	vbo_color = engine->get_vbo_color();

	// Initialize and fill vbo_attributes
	std::cout << std::endl << " Initializing and linking vbo_attributes " << std::endl;
	vbo_attributes = engine->get_vbo_attributes();
#endif

	// Start GLUT loop
	glutMainLoop();

	return 0;

}// End of main function

//----------------------------------------

void init_trackers() {
	// Initialize colored trackers
	if ( TRACKER_COUNT > 0 ) {
		if ( points == nil ) points = new float[3*TRACKER_COUNT];
		if ( ranges == nil ) ranges = new float[3*TRACKER_COUNT];
		if ( colors == nil ) colors = new float[3*TRACKER_COUNT];
	}
	MTRand rand_gen(time(0));
	for ( int k = 0; k < TRACKER_COUNT; ++k ) {
		points[3*k+0] = -1.0 + 2.0*rand_gen.rand();
		points[3*k+1] = -1.0 + 2.0*rand_gen.rand();
		points[3*k+2] = -1.0 + 2.0*rand_gen.rand();
		ranges[3*k+0] = 0.05 + 0.15*rand_gen.rand();
		ranges[3*k+1] = 0.05 + 0.15*rand_gen.rand();
		ranges[3*k+2] = 0.05 + 0.15*rand_gen.rand();
		colors[3*k+0] = rand_gen.rand();
		colors[3*k+1] = rand_gen.rand();
		colors[3*k+2] = rand_gen.rand();
	}
#ifndef CPU_MODE
	resetCuda(points, ranges, colors, 3*TRACKER_COUNT);
#endif
}

//----------------------------------------

void load_gslib() {
	// Clean space
	if ( data_vector != nil ) {
		delete [] data_vector;
		data_vector = nil;
	}

	// Load GSLIB file
	std::list<std::string> property_names;
	if ( gslib == 0 ) gslib_file = "u_speed";
	else if ( gslib == 1 ) gslib_file = "channel";
	else if ( gslib == 2 ) gslib_file = "vortex";
	else if ( gslib == 3 ) gslib_file = "vortex_spiral";
	else if ( gslib == 4 ) gslib_file = "producer";
	std::string path = "../data/gslib/" + gslib_file + ".gslib";
	data_vector = GSlib::load_file<float>(nb_props, sizeu, sizev, sizew, property_names, path.c_str());
	++gslib;
	if ( gslib > 4 ) gslib = 0;
	size_vector = nb_props*sizeu*sizev*sizew;
#ifndef CPU_MODE
	fill_vbo_vector(data_vector, size_vector, nb_props);
	vbo_vector_cuda = get_vbo_vector_cuda();
	if ( data_vector != nil ) {
		delete [] data_vector;
		data_vector = nil;
	}
#endif
}

//----------------------------------------

void reset_engine() {
	// Clean space
	float opacity = 0.15;
	float local_radius = PARTICLES_LOCAL_RADIUS;
	if ( engine != nil ) {
		opacity = engine->get_engine_opacity();
		local_radius = engine->get_engine_local_radius();
		delete engine;
	}

	// Update index
	++pindex;
	if ( pindex > 5 ) pindex = 0;

	// Reset particle engine
	engine = new ParticleEngine(pcount[pindex]);
	engine->set_engine_width_and_height(width, height);
	engine->set_engine_opacity(opacity);
	engine->set_engine_local_radius(local_radius);
}

//----------------------------------------

#ifndef CPU_MODE
void fill_vbo_vector(float* data_vector, int size_vector, int nb_props) {
	// Allocate and assign three Vertex Buffer Objects to our handle
    glGenBuffers(1, vbo_vector);

	// Bind our VBO as being the active buffer and storing vectors */
    glBindBuffer(GL_ARRAY_BUFFER, vbo_vector[0]);

	// Filling the float* that will be used to fill the vbo
	int vector_array_size = 0;
	float* vector_array_tmp;
	if ( nb_props == 3 ) {
		vector_array_tmp = new float[size_vector];
		vector_array_tmp = data_vector;
		vector_array_size = size_vector;
	}
	if ( nb_props == 2 ) { // If vectors have only two values, the third one is set to "0" in the float* that will be used to fill the vector_vbo
		int nb_cells = size_vector/nb_props;
		vector_array_size = 3*nb_cells;
		vector_array_tmp = new float[vector_array_size];
		for ( int i = 0 ; i < nb_cells ; ++i ){
			vector_array_tmp[i*3] = data_vector[i*nb_props];
			vector_array_tmp[i*3+1] = data_vector[i*nb_props+1];
			vector_array_tmp[i*3+2] = 0 ;
		}
	}
	if ( nb_props == 1 ) {
		std::cout << " nb_props error in data_vector " <<std::endl;
		return;
	}
	glBufferData(GL_ARRAY_BUFFER, vector_array_size*sizeof(float), vector_array_tmp, GL_STATIC_DRAW); // Allocation de la place en memoire du GPU

    // Specify that our coordinate data is going into attribute index 0, and contains 3 floats per vertex
    glVertexAttribPointer((GLuint)3, 3, GL_FLOAT, GL_FALSE, 0, 0);
	glBindBuffer(GL_ARRAY_BUFFER, 0);
}
#endif
