//=========================================================
//          Uncertainty visualization example
//=========================================================
//
// This code demonstrates the use of GLSL shaders to convey
// a sense of uncertainty in geological issues.
//
// Two different uncertainty visualization methods are shown:
//     (i)  Uncertainty mapped to texture intensity
//     (ii) Uncertainty mapped to blur intensity
//
// Authors: Thomas Viard, Guillaume Caumon and Bruno Lvy
// Email: viard@gocad.org
//
// This software contains source code provided by NVIDIA
// Corporation (code for viewpoint manipulation).

#include <iostream>
#include <fstream>
#include <vector>
#include <map>
#include <string>
#include <sstream>
#include <list>

#include <GL/glew.h>
#include <GL/glut.h>

#include <nvGlutManipulators.h>

using std::vector;
using std::map;
using std::string;



//==================================================================
//                        Global variables                        //
//==================================================================

// Enumerator listing available options
enum options_define {
    OPTION_TEXTURE,
    OPTION_BLUR,
    OPTION_COUNT
};

// List of available options ad key map
bool options[OPTION_COUNT];
map<char, options_define> optionKeyMap;

// Text display list
GLuint textDList = 0;

// Manipulator
nv::GlutExamine manipulator;

// Initial size of the window
int width = 512;
int height = 512;

// Shader programs identifiers
GLuint textureProgram = 0;
GLuint blurProgram = 0;
GLuint normalProgram = 0;
GLuint cmapProgram = 0;
GLuint currentProgram;

// Shaders identifiers
GLuint vert = 0;
GLuint fragNormal = 0;
GLuint fragTexture = 0;
GLuint fragBlur = 0;
GLuint fragCmap = 0;

// Lookup texture
GLuint lookup = 0;
int u_max = 0;
int v_max = 0;
int field = 0;
int ufield = 0;
std::string fields[4];
int nb_props = 0;

// Texture pattern
GLuint pattern = 0;

// Colormap
GLuint cmap = 0;

// Maximum texture intensity
float max_intensity = 1.0f;

// Blur radius
int blur_radius = 3;

// Intensity reverse
int reverse = 0;

// Global state
bool ok = true;

// Vertex shader
const char* vertex_shader_str = {
	"// Main function of the vertex shader\n"
	"void main() {\n"
	"	// Pass the texture coordinates to the fragment shader\n"
	"	gl_TexCoord[0] = gl_MultiTexCoord0; // Lookup texture coordinates\n"
	"	gl_TexCoord[1] = gl_MultiTexCoord1; // Texture pattern coordinates\n"
	"\n"
	"	// Transform the vertex position\n"
	"	gl_Position = ftransform();\n"
	"}\n"
};

// Colormap fragment shader
const char* fragment_shader_cmap_str = {
	"// Colormap\n"
	"uniform sampler1D colormap;\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// Main function of the fragment shader\n"
	"void main() {\n"
	"	// Get the color of the colormap at current pixel\n"
	"	gl_FragColor = texture1D(colormap,gl_TexCoord[0].s);\n"
	"}\n"
};

// Normal fragment shader
const char* fragment_shader_normal_str = {
	"// Color map used for the primary variable\n"
	"uniform sampler1D colormap;\n"
	"\n"
	"// Lookup that contains both the primary variable and the uncertainty values\n"
	"uniform sampler2D lookup;\n"
	"\n"
	"// Color field in the lookup\n"
	"uniform int field;\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// Main function of the fragment shader\n"
	"void main() {\n"
	"	// Access the primary variable, stored in a texture lookup\n"
	"	vec4 lookup_values = texture2D(lookup, gl_TexCoord[0].st);\n"
	"	float values[2];\n"
	"	values[0] = lookup_values.x;\n"
	"	values[1] = lookup_values.y;\n"
	"	float value = values[field];\n"
	"\n"
	"	// Get the color of the background property at current pixel\n"
	"	gl_FragColor = texture1D(colormap,value);\n"
	"}\n"
};

// Texture-based uncertainty visualization fragment shader
const char* fragment_shader_texture_str = {
	"// Color map used for the primary variable\n"
	"uniform sampler1D colormap;\n"
	"\n"
	"// Texture pattern for uncertainty display\n"
	"uniform sampler2D texture_pattern;\n"
	"\n"
	"// Lookup that contains both the primary variable and the uncertainty values\n"
	"uniform sampler2D lookup;\n"
	"\n"
	"// Maximal intensity of the texture in the uncertainty visualization\n"
	"// This variable can be changed to customize interferences. It is assumed to be in range [0,1]\n"
	"uniform float max_intensity;\n"
	"\n"
	"// Color field in the lookup\n"
	"uniform int field;\n"
	"\n"
	"// Uncertainty field in the lookup\n"
	"uniform int ufield;\n"
	"\n"
	"// Reverse intensity function\n"
	"uniform bool reverse;\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// Function used to map local uncertainty to texture intensity (linear by default)\n"
	"float texture_intensity(float uncertainty) {\n"
	"	if ( reverse ) return 1.0 - uncertainty;\n"
	"	return uncertainty;\n"
	"}\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// Main function of the fragment shader\n"
	"void main() {\n"
	"// Step 1: data access\n"
	"	// Access the primary variable and its associated uncertainty, stored in a texture lookup\n"
	"	vec4 lookup_values = texture2D(lookup, gl_TexCoord[0].st);\n"
	"	float values[4];\n"
	"	values[0] = lookup_values.x;\n"
	"	values[1] = lookup_values.y;\n"
	"	values[2] = lookup_values.z;\n"
	"	values[3] = lookup_values.w;\n"
	"\n"
	"// Step 2: translate into colors\n"
	"	// Get the color of the background property at current pixel\n"
	"	vec4 color = texture1D(colormap,values[field]);\n"
	"\n"
	"	// Get the color of the texture pattern at current pixel\n"
	"	vec4 tex_color = texture2D(texture_pattern, gl_TexCoord[1].st);\n"
	"\n"
	"// Step 3: blend colors depending on local uncertainty\n"
	"	// Get the uncertainty-dependent texture intensity\n"
	"	float intensity = max_intensity * tex_color.a * texture_intensity(values[ufield]);\n"
	"\n"
	"	// Blend RGB components of the colors based on the local uncertainty degree\n"
	"	gl_FragColor.rgb = ( 1.0 - intensity ) * color.rgb + intensity * tex_color.rgb;\n"
	"	gl_FragColor.a = color.a;\n"
	"}\n"
};

// Blur-based uncertainty visualization fragment shader
const char* fragment_shader_blur_str = {
	"// Color map used for the primary variable\n"
	"uniform sampler1D colormap;\n"
	"\n"
	"// Lookup that contains both the primary variable and the uncertainty values\n"
	"uniform sampler2D lookup;\n"
	"\n"
	"// Size of the slice; first component is for local U axis and second for local V axis\n"
	"uniform vec2 slice_size;\n"
	"\n"
	"// Blur radius\n"
	"uniform int blur_radius;\n"
	"\n"
	"// Color field in the lookup\n"
	"uniform int field;\n"
	"\n"
	"// Uncertainty field in the lookup\n"
	"uniform int ufield;\n"
	"\n"
	"// Reverse intensity function\n"
	"uniform bool reverse;\n"
	"\n"
	"// Value of Pi\n"
	"#define M_PI 3.14159\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// Gaussian function (the mean is assumed to be 0)\n"
	"float gaussian(float x, float sigma) {\n"
	"	return exp(-x*x/(2.0*sigma*sigma)) / (sigma*sqrt(2.0*M_PI));\n"
	"}\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// This function finds the color of the blurred pixel (assumes Gaussian blur with given blur radius)\n"
	"vec4 blur_pixel(int radius, vec2 coords) {\n"
	"// Initialize variables\n"
	"	float sum = 0.0;                                           // Sum of the pixel contributions\n"
	"	vec4 blurred_color = vec4(0.0);                            // Final blurred color\n"
	"	int nb_samples = 3;                                        // Number of samples per texel\n"
	"	vec2 offset = vec2(1.0) / (float(nb_samples)*slice_size);  // Texture coordinates offset at each step of the loop\n"
	"\n"
	"// Loop over all texels that contribute to the blurred pixel (contribution beyond 3 times the radius is neglected)\n"
	"	for ( int k = -3*nb_samples*radius; k < 3*nb_samples*radius; ++k ) {\n"
	"		for ( int l = -3*radius; l < 3*radius; ++l ) {\n"
	"			// Get global pixel contribution\n"
	"			float contribution = gaussian(float(k),float(radius)) * gaussian(float(l),float(radius));\n"
	"\n"
	"			// Offset the texture coordinates to find the pixel color\n"
	"			vec4 cur_values = texture2D(lookup, coords + vec2(k,l)*offset);\n"
	"			float values[4];\n"
	"			values[0] = cur_values.x;\n"
	"			values[1] = cur_values.y;\n"
	"			values[2] = cur_values.z;\n"
	"			values[3] = cur_values.w;\n"
	"			float cur_value = values[field];\n"
	"			vec4 cur_color = texture1D(colormap, cur_value);\n"
	"\n"
	"			// Add contribution to the blurred color\n"
	"			blurred_color += contribution*cur_color;\n"
	"			sum += contribution;\n"
	"		}\n"
	"	}\n"
	"\n"
	"// Output result\n"
	"	// Return the normalized blurred pixel color\n"
	"	return blurred_color / sum;\n"
	"}\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// Function used to map local uncertainty to blur intensity (linear by default)\n"
	"float blur_intensity(float uncertainty) {\n"
	"	if ( reverse ) return uncertainty;\n"
	"	return 1.0 - uncertainty;\n"
	"}\n"
	"\n"
	"//=============================================================\n"
	"\n"
	"// Main function of the fragment shader\n"
	"void main() {\n"
	"	// Get sharp pixel color\n"
	"	vec4 lookup_values = texture2D(lookup, gl_TexCoord[0].st);\n"
	"	float values[4];\n"
	"	values[0] = lookup_values.x;\n"
	"	values[1] = lookup_values.y;\n"
	"	values[2] = lookup_values.z;\n"
	"	values[3] = lookup_values.w;\n"
	"	vec4 color = texture1D(colormap, values[field]);\n"
	"\n"
	"	// Get blurred pixel color\n"
	"	vec4 blur_color = blur_pixel(blur_radius, gl_TexCoord[0].st);\n"
	"\n"
	"	// Get the uncertainty-dependent blur intensity\n"
	"	float intensity = blur_intensity(values[ufield]);\n"
	"\n"
	"	// Blend the colors based on the local uncertainty degree\n"
	"	gl_FragColor = ( 1.0 - intensity ) * color + intensity * blur_color;\n"
	"}\n"
};



//==================================================================
//                           Functions                            //
//==================================================================

void setup_1d_texture(GLuint& texture, const char* filename);
void setup_2d_texture_from_raw_file(GLuint& texture, const char* filename, int& umax, int& vmax, bool repeat);
void setup_2d_texture_from_gslib_file(GLuint& texture, const char* filename, int& umax, int& vmax, bool repeat);

// Initialize OpenGL state
void init_opengl() {
	// Init GLEW
	glewInit();

	// Check GPU supported features
	if (!glewIsSupported( "GL_VERSION_2_0 GL_ARB_fragment_program" )) {
		printf( "Error: failed to get minimal extensions for demo\n");
		printf( "This sample requires:\n");
		printf( "  OpenGL version 2.0\n");
		printf( "  GL_ARB_fragment_program\n");
		exit(-1);
	}

	// Setup default OpenGL state
	glEnable(GL_DEPTH_TEST);
	glClearColor(0.0, 0.0, 0.0, 1.0);
	glDepthFunc(GL_LEQUAL);

	// Generate display lists for ASCII characters
	textDList = glGenLists(128);
	for ( int k = 0; k < 128; ++k) {
		glNewList( k + textDList, GL_COMPILE);
		glutBitmapCharacter(  GLUT_BITMAP_HELVETICA_12, k );
		glEndList();
	}
	glListBase(textDList);

	// Setup other textures
	setup_1d_texture(cmap, "../data/colormap.dat");
	int uc, vc;
	setup_2d_texture_from_gslib_file(lookup, "../data/data.gslib", u_max, v_max, false);
	setup_2d_texture_from_raw_file(pattern, "../data/canvas.dat", uc, vc, true);
}

//----------------------------------------

bool check_compiling(GLuint shader);
bool check_linking(GLuint program);

// Setup shader program
void setup_shader(GLuint& program, GLuint vert_shader, const char* vert_source, GLuint frag_shader, const char* frag_source) {
	// Check program existence
	if ( program != 0 ) return;
	program = glCreateProgram();

	// Check that vertex and fragment shaders are compiled and attached to the program
	if ( vert_shader == 0 ) {
		vert_shader = glCreateShader(GL_VERTEX_SHADER);
		glShaderSource(vert_shader, 1, &vert_source, NULL);
		glCompileShader(vert_shader);
		ok &= check_compiling(vert_shader);
		glAttachShader(program, vert_shader);
	}
	if ( frag_shader == 0 ) {
		frag_shader = glCreateShader(GL_FRAGMENT_SHADER);
		glShaderSource(frag_shader, 1, &frag_source, NULL);
		glCompileShader(frag_shader);
		ok &= check_compiling(frag_shader);
		glAttachShader(program, frag_shader);
	}

	// Link program
	glLinkProgram(program);
	ok &= check_linking(program);
}

// Check whether the shader was correctly compiled
bool check_compiling(GLuint shader) {
	// Check compile status
    int success;
    glGetShaderiv(shader, GL_COMPILE_STATUS, &success);

	// Print error message in case of failure
	if(success == GL_FALSE) {
		// Get error log
        int InfoLogSize;
        glGetShaderiv(shader, GL_INFO_LOG_LENGTH, &InfoLogSize);
        char* Buffer = new char[InfoLogSize];
		glGetShaderInfoLog(shader, InfoLogSize, NULL, Buffer);

		// Print error log
		std::cout << Buffer << std::endl;
        delete[] Buffer;

		// Return failure
		return false;
	}
    return true;
}

// Check whether the shader was correctly linked
bool check_linking(GLuint program) {
	// Check link status
    int success;
    glGetProgramiv(program, GL_LINK_STATUS, &success);

	// Print error message in case of failure
	if(success == GL_FALSE) {
		// Get error log
        int InfoLogSize;
        glGetProgramiv(program, GL_INFO_LOG_LENGTH, &InfoLogSize);
        char* Buffer = new char[InfoLogSize];
		glGetProgramInfoLog(program, InfoLogSize, NULL, Buffer);

		// Print error log
		std::cout << Buffer << std::endl;
        delete[] Buffer;

		// Return failure
		return false;
	}

	// Return success
    return true;
}

//----------------------------------------

// Setup the normal shader program
void setup_normal_shader() {
	setup_shader(normalProgram, vert, vertex_shader_str, fragNormal, fragment_shader_normal_str);
	glUseProgram(normalProgram);
	currentProgram = normalProgram;
}

//----------------------------------------

// Setup the texture shader program
void setup_texture_shader() {
	setup_shader(textureProgram, vert, vertex_shader_str, fragTexture, fragment_shader_texture_str);
	glUseProgram(textureProgram);
	currentProgram = textureProgram;
}

//----------------------------------------

// Setup the blur shader program
void setup_blur_shader() {
	setup_shader(blurProgram, vert, vertex_shader_str, fragBlur, fragment_shader_blur_str);
	glUseProgram(blurProgram);
	currentProgram = blurProgram;
}

//----------------------------------------

// Setup the colormap shader program
void setup_cmap_shader() {
	setup_shader(cmapProgram, vert, vertex_shader_str, fragCmap, fragment_shader_cmap_str);
	glUseProgram(cmapProgram);
}

//----------------------------------------

// Setup a 1D texture
void setup_1d_texture(GLuint& texture, const char* filename) {
	if ( texture == 0 ) glGenTextures(1, &texture);

	// Access raw file
	std::ifstream in(filename);
	if ( !in.good() ) {
		std::cout << "Input file error" << std::endl;
		return;
	}

	// Get texture size
	int indexmax;
	in >> indexmax;

	// Generate float array
	float* tex_array = new float[4*indexmax];

	// Iterate over the texels
	for ( int index = 0; index < 4*indexmax; ++index ) {
		// Get texel color
		in >> tex_array[index];
	}

	// Reset pixel storage
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	// Init texture
	glEnable(GL_TEXTURE_1D);
	glBindTexture( GL_TEXTURE_1D, texture );
	glTexImage1D( GL_TEXTURE_1D, 0, GL_RGBA, indexmax, 0, GL_RGBA, GL_FLOAT, tex_array );
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_1D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	// Reset texture environment
	glBindTexture( GL_TEXTURE_1D, 0 );
	delete [] tex_array;
}

//----------------------------------------

// Setup a 2D texture
void setup_2d_texture_from_raw_file(GLuint& texture, const char* filename, int& umax, int& vmax, bool repeat) {
	if ( texture == 0 ) glGenTextures(1, &texture);

	// Access raw file
	std::ifstream in(filename);
	if ( !in.good() ) {
		std::cout << "Input file error" << std::endl;
		return;
	}

	// Get texture size
	in >> umax >> vmax;

	// Generate float array
	float* tex_array = new float[4*umax*vmax];

	// Iterate over the texels
	long index = 0;
	for ( int u = 0; u < umax; ++u ) {
		for ( int v = 0; v < vmax; ++v ) {
			// Get texel color
			in >> tex_array[index] >> tex_array[index+1] >> tex_array[index+2] >> tex_array[index+3];
			index += 4;
		}
	}

	// Reset pixel storage
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	// Init texture
	glEnable(GL_TEXTURE_2D);
	glBindTexture( GL_TEXTURE_2D, texture );
	glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, umax, vmax, 0, GL_RGBA, GL_FLOAT, tex_array );
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	if ( repeat ) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	}
	else {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	}
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	// Reset texture environment
	glBindTexture( GL_TEXTURE_2D, 0 );
	delete [] tex_array;
}

//----------------------------------------

// Split a string into a set of substrings
void string_split(std::list<std::string>& result, const std::string& str) {
	result.clear();
	std::istringstream iss(str);
	while (iss) {
		std::string sub;
		iss >> sub;
		if ( !sub.empty() ) result.push_back(sub);
	}
}

//----------------------------------------

// Convert a string into any numeric type
template <class T>
T string_convert(const std::string& str) {
	T result;
	std::istringstream iss(str);
	iss >> result;
	return result;
}

//----------------------------------------

// Setup a 2D texture
void setup_2d_texture_from_gslib_file(GLuint& texture, const char* filename, int& umax, int& vmax, bool repeat) {
	if ( texture == 0 ) glGenTextures(1, &texture);

	// Access raw file
	std::ifstream in(filename);
	if ( !in.good() ) {
		std::cout << "Input file error" << std::endl;
		return;
	}

	// Get texture size
	{{
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		if ( list.size() < 2 ) {
			std::cout << "Cannot read U and V sizes. First line of input file must be \"U V\", where U and V are integers." << std::endl;
			return;
		}
		std::list<std::string>::const_iterator it = list.begin();
		umax = string_convert<int>(*it);
		++it;
		vmax = string_convert<int>(*it);
	}}

	// Get number of properties
	int actual_nb_props = 0;
	{{
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		std::list<std::string>::const_iterator it = list.begin();
		actual_nb_props = string_convert<int>(*it);
		nb_props = actual_nb_props;
		if ( nb_props > 4 ) {
			std::cout << "Warning - the GSLIB file contains more than 4 properties. Only the four first properties will be kept." << std::endl;
			nb_props = 4;
		}
		if ( nb_props > 1 ) ufield = 1;
	}}

	// Get property names
	for ( int k = 0; k < actual_nb_props; ++k ) {
		char lineStr[1024];
		in.getline(lineStr, 1024);
		std::string line(lineStr);
		std::list<std::string> list;
		string_split(list, line);
		std::list<std::string>::const_iterator it = list.begin();
		if ( k < 4 ) fields[k] = it->c_str();
	}

	// Generate float array
	float* tex_array = new float[4*umax*vmax];

	// Iterate over the texels
	long index = 0;
	for ( int u = 0; u < umax; ++u ) {
		for ( int v = 0; v < vmax; ++v ) {
			// Get line
			char lineStr[1024];
			in.getline(lineStr, 1024);
			std::string line(lineStr);

			// Split line
			std::list<std::string> list;
			string_split(list, line);

			// Get texel value
			std::list<std::string>::const_iterator it = list.begin();
			int i = 0;
			while ( ( i < nb_props ) && ( it != list.end() ) ) {
				tex_array[index+i] = string_convert<float>(*it);
				++i; ++it;
			}
			for ( int k = i; k < 4; ++k ) tex_array[index+k] = 0.0f;
			index += 4;
		}
	}

	// Reset pixel storage
	glPixelStorei(GL_UNPACK_ALIGNMENT, 1);

	// Init texture
	glEnable(GL_TEXTURE_2D);
	glBindTexture( GL_TEXTURE_2D, texture );
	glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, umax, vmax, 0, GL_RGBA, GL_FLOAT, tex_array );
	glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
	if ( repeat ) {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	}
	else {
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
		glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	}
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	// Reset texture environment
	glBindTexture( GL_TEXTURE_2D, 0 );
	delete [] tex_array;
}

//----------------------------------------

// Clean shader programs
void cleanup() {
	// Clean shaders
	if ( vert != 0 ) glDeleteShader(vert);
	if ( fragNormal != 0 ) glDeleteShader(fragNormal);
	if ( fragTexture != 0 ) glDeleteShader(fragTexture);
	if ( fragBlur != 0 ) glDeleteShader(fragBlur);
	if ( fragCmap != 0 ) glDeleteShader(fragCmap);

	// Clean programs
	if ( normalProgram != 0 ) glDeleteProgram(normalProgram);
	if ( textureProgram != 0 ) glDeleteProgram(textureProgram);
	if ( blurProgram != 0 ) glDeleteProgram(blurProgram);
	if ( cmapProgram != 0 ) glDeleteProgram(cmapProgram);

	// Clean display lists
	if ( textDList != 0 ) glDeleteLists(textDList, 128);

	// Clean textures
	if ( lookup ) glDeleteTextures(1, &lookup);
	if ( pattern ) glDeleteTextures(1, &pattern);
	if ( cmap ) glDeleteTextures(1, &cmap);

	// Clean fields string
	if ( fields ) delete [] fields;
}

//----------------------------------------

// Display data
void display() {
	// Clear previous display
	{{
		glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	}}

    // Render the user interface
	{{
		// Setup OpenGL state
		glMatrixMode(GL_PROJECTION);
		glPushMatrix();
		glLoadIdentity();
		glOrtho( 0.0, double(width), 0.0, double(height), 0.0, 1.0);
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();

		// Draw colormap
		setup_cmap_shader();
		glActiveTexture( GL_TEXTURE0 );
		glBindTexture(GL_TEXTURE_1D, cmap);
		glColor3f(1.0f, 1.0f, 1.0f);
		glBegin(GL_QUADS);
		{
			glTexCoord1i(1); glVertex3i( width-100, height-25, 0);
			glTexCoord1i(0); glVertex3i( width-10,  height-25, 0);
			glTexCoord1i(0); glVertex3i( width-10,  height-10, 0);
			glTexCoord1i(1); glVertex3i( width-100, height-10, 0);
		}
		glEnd();
		glBindTexture(GL_TEXTURE_1D, 0);
		glUseProgram(0);
		glBegin(GL_LINE_STRIP);
		{
			glVertex3f( width-100, height-25, 0);
			glVertex3f( width-10,  height-25, 0);
			glVertex3f( width-10,  height-10, 0);
			glVertex3f( width-100, height-10, 0);
			glVertex3f( width-100, height-25, 0);
		}
		glEnd();

		// Render a text overlay
		char size[256];
		glColor3f(1.0f, 1.0f, 1.0f);
		if ( options[OPTION_TEXTURE] ) sprintf( size, "Texture-based uncertainty visualization");
		else if ( options[OPTION_BLUR] ) sprintf( size, "Blur-based uncertainty visualization");
		else sprintf( size, "No uncertainty visualization");
		glRasterPos2f( 10.0f, height - 20.0f);
		glCallLists( (GLsizei)strlen(size), GL_UNSIGNED_BYTE, size);
		glBindTexture( GL_TEXTURE_2D, 0);

		// Reset matrices
		glMatrixMode(GL_PROJECTION);
		glPopMatrix();
		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
	}}

	// Draw the data
	{{
		// Setup OpenGL state
		glMatrixMode(GL_MODELVIEW);
		glPushMatrix();
		glLoadIdentity();

		// Apply manipulator
		manipulator.applyTransform();

		// Setup shader
		if ( options[OPTION_TEXTURE] ) {
			setup_texture_shader();
			glActiveTexture( GL_TEXTURE2 );
			glBindTexture(GL_TEXTURE_2D, pattern);
			glUniform1i(glGetUniformLocation(currentProgram, "texture_pattern"), 2);
			glUniform1f(glGetUniformLocation(currentProgram, "max_intensity"), max_intensity);
		}
		else if ( options[OPTION_BLUR] ) {
			setup_blur_shader();
			glUniform2f(glGetUniformLocation(currentProgram, "slice_size"), u_max, v_max);
			glUniform1i(glGetUniformLocation(currentProgram, "blur_radius"), blur_radius);
		}
		else setup_normal_shader();

		// Setup textures
		glActiveTexture( GL_TEXTURE0 );
		glBindTexture(GL_TEXTURE_2D, lookup);
		glUniform1i(glGetUniformLocation(currentProgram, "lookup"), 0);
		glActiveTexture( GL_TEXTURE1 );
		glBindTexture(GL_TEXTURE_1D, cmap);
		glUniform1i(glGetUniformLocation(currentProgram, "colormap"), 1);

		// Setup field and reverse intensity function
		glUniform1i(glGetUniformLocation(currentProgram, "field"), field);
		glUniform1i(glGetUniformLocation(currentProgram, "ufield"), ufield);
		glUniform1i(glGetUniformLocation(currentProgram, "reverse"), reverse);

		// Draw textured quad
		glColor3f(1.0f, 1.0f, 1.0f);
		glBegin(GL_QUADS);
		{
			float repeat = 30.0f;
			glMultiTexCoord2f( GL_TEXTURE0, 0.0f, 0.0f );
			glMultiTexCoord2f( GL_TEXTURE1, 0.0f, 0.0f );
			glVertex2f(-1, -1);
			glMultiTexCoord2f( GL_TEXTURE0, 1.0f, 0.0f );
			glMultiTexCoord2f( GL_TEXTURE1, repeat, 0.0f );
			glVertex2f( 1, -1);
			glMultiTexCoord2f( GL_TEXTURE0, 1.0f, 1.0f );
			glMultiTexCoord2f( GL_TEXTURE1, repeat, repeat );
			glVertex2f( 1,  1);
			glMultiTexCoord2f( GL_TEXTURE0, 0.0f, 1.0f );
			glMultiTexCoord2f( GL_TEXTURE1, 0.0f, repeat );
			glVertex2f(-1,  1);
		}
		glEnd();

		// Remove textures
		if ( options[OPTION_TEXTURE] ) {
			glActiveTexture( GL_TEXTURE2 );
			glBindTexture(GL_TEXTURE_2D, 0);
		}
		glActiveTexture( GL_TEXTURE1 );
		glBindTexture(GL_TEXTURE_1D, 0);
		glActiveTexture( GL_TEXTURE0 );
		glBindTexture(GL_TEXTURE_2D, 0);

		// Reset modelview matrix
		glPopMatrix();

		// Remove program
		glUseProgram(0);
	}}

	glutSwapBuffers();
}

//----------------------------------------

// Reset the display when idling
void idle() {
	glutPostRedisplay();
}

//----------------------------------------

void help() {
	// Clear previous display
	if ( ok == false ) return;
	system("cls");

	// Help on commands
	printf( "\n Uncertainty visualization commands\n\n");
	printf( "  ---------------------------------------------\n");
	printf( "     q/[ESC]    - Quit the application\n");
	printf( "  ---------------------------------------------\n");
	printf( "        t       - Toggle texture-based uncertainty visualization\n");
	printf( "        b       - Toggle blur-based uncertainty visualization\n");
	printf( "       +/-      - [Texture mode] Increase/decrease maximum texture intensity\n");
	printf( "       +/-      -  [Blur mode]   Increase/decrease blur radius\n");
	printf( "        r       - Toggle reversed intensity function\n");
	printf( "  ---------------------------------------------\n");
	printf( "        v       - Switch color to next variable\n");
	printf( "        u       - Switch uncertainty to next variable\n");
	printf( "  ---------------------------------------------\n");
	printf( " [MOUSE LEFT]   - Rotate the view point\n");
	printf( " [MOUSE RIGHT]  - Zoom\n");
	printf( " [MOUSE MIDDLE] - Move the view point\n");
	printf( "  ---------------------------------------------\n\n");

	// Help on current state
	if ( options[OPTION_TEXTURE] ) {
		printf( "State: Texture-based uncertainty visualization\n");
	}
	else if ( options[OPTION_BLUR] ) {
		printf( "State: Blur-based uncertainty visualization\n");
	}
	else {
		printf( "State: No uncertainty visualization\n");
	}
	printf( "  -> Color       <=> %s\n", fields[field].c_str());
	printf( "  -> Uncertainty <=> %s\n", fields[ufield].c_str());
	if ( options[OPTION_TEXTURE] ) {
		printf( "  -> Texture max intensity = %f\n", max_intensity);
		if ( reverse == 0 ) printf( "  -> Linear intensity function\n");
		else printf( "  -> Reversed intensity function\n");
	}
	else if ( options[OPTION_BLUR] ) {
		printf( "  -> Blur radius = %d\n", blur_radius);
		if ( reverse == 0 ) printf( "  -> Linear intensity function\n");
		else printf( "  -> Reversed intensity function\n");
	}
	printf( "\n");
}

//----------------------------------------

// Function called when a key is pressed
void key(unsigned char k, int x, int y) {
	// Low character
	k = tolower(k);

	// Find corresponding option, if any
	if (optionKeyMap.find(k) != optionKeyMap.end()) {
		options[optionKeyMap[k]] = !options[optionKeyMap[k]];
		if ( k == 'b' ) options[optionKeyMap['t']] = false;
		else if ( k == 't' ) options[optionKeyMap['b']] = false;
	}

	if(k=='+') {
		if ( options[OPTION_TEXTURE] ) {
			max_intensity += 0.1f;
			if ( max_intensity > 1.0f ) max_intensity = 1.0f;
		}
		else if ( options[OPTION_BLUR] ) {
			++blur_radius;
		}
	}
	if(k=='-') {
		if ( options[OPTION_TEXTURE] ) {
			max_intensity -= 0.1f;
			if ( max_intensity < 0.0f ) max_intensity = 0.0f;
		}
		else if ( options[OPTION_BLUR] ) {
			if ( blur_radius > 1 ) --blur_radius;
		}
	}

	// Toggle color display
	if (k=='v') {
		field = (1+field)%nb_props;
	}

	// Toggle uncertainty display
	if (k=='u') {
		ufield = (1+ufield)%nb_props;
	}

	// Toggle uncertainty-color display
	if (k=='r') {
		if ( options[OPTION_TEXTURE] || options[OPTION_BLUR] ) {
			reverse = 1-reverse;
		}
	}

	// Deal with quit key
	if(k==27 || k=='q') {
		cleanup();
		exit(0);
	}

	// Reset display
	help();
	glutPostRedisplay();
}

//----------------------------------------

// Resize the window
void resize(int w, int h) {
	// Check height
	if (h == 0) h = 1;

	// Update view parameters
	glViewport(0, 0, w, h);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(60.0, (GLfloat)w/(GLfloat)h, 0.1, 100.0);
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	// Update manipulator
	manipulator.reshape(w, h);
	width = w;
	height = h;
}

//----------------------------------------

// Function called when the mouse is pressed
void mouse(int button, int state, int x, int y) {
	manipulator.mouse(button, state, x, y);
}

//----------------------------------------

// Motion function
void motion(int x, int y) {
	manipulator.motion(x, y);
}

//----------------------------------------

// Main function
int main(int argc, char **argv)
{
	// Initialize GLUT and create window
	glutInit(&argc, argv);
	glutInitWindowSize( width, height);
	glutInitDisplayMode(GLUT_DOUBLE | GLUT_DEPTH | GLUT_RGB);
	glutCreateWindow("Uncertainty visualization");

	// Initialize miscellaneous variables
	for ( int k = 0; k < 4; ++k ) fields[k] = "None";

	// Initialize OpenGL
	init_opengl();

	// Setup manipulator
	manipulator.setDollyActivate( GLUT_RIGHT_BUTTON );
	manipulator.setPanActivate( GLUT_MIDDLE_BUTTON );
	manipulator.setDollyPosition( -2.0f);

	// Setup GLUT important functions
	glutDisplayFunc(display);
	glutMouseFunc(mouse);
	glutMotionFunc(motion);
	glutIdleFunc(idle);
	glutKeyboardFunc(key);
	glutReshapeFunc(resize);

	// Setup key/options map
	optionKeyMap['t'] = OPTION_TEXTURE;
	options[OPTION_TEXTURE] = false;
	optionKeyMap['b'] = OPTION_BLUR;
	options[OPTION_BLUR] = false;

	// Display user commands
	if ( nb_props <= 0 ) {
		std::cout << "Error - no property was read." << std::endl;
		system("PAUSE");
		return 1;
	}
	help();

	// Start GLUT loop
	glutMainLoop();
	return 0;
}
